package org.ourfilesystem.core;

/*
 *  
    OurFileSystem is a peer2peer file sharing program.
    Copyright (C) 2012  Robert Gass

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 */

import java.io.File;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

import org.ourfilesystem.db.LocalFileReference;
import org.ourfilesystem.db.LocalPost;
import org.ourfilesystem.db.Peer;

public interface CoreUserInterface {
	
	/**
	 * This is to get information about current connections this
	 * peer has.  The keys are the peers we're currently connected to
	 * the values are the number of failures that peers has had. 
	 * @return
	 */
	public HashSet<Peer> getConnections();
	
	/**
	 * Request a file be downloaded by its digest.
	 * @param digest - The digest of the file to download.
	 */
	public void downloadFile(Object digest);
	
	/**
	 * Request that we update our peer list.  It will contact 
	 * other nodes to ask for their list of peers.  If a new
	 * peer has joined the network we would validate its signature
	 * and add it to our database.
	 */
	public void updatePeers();
	
	/**
	 * Request that we update posts.  This will request new posts
	 * from other nodes and ask for missing sequence numbers.
	 * New posts have their signature checked and are then added
	 * to our database.
	 */
	public void updatePosts();
	
	/**
	 * The local operator of this node wishes to add a new post.
	 * @param message - The message File.
	 * @param fileref - An optional file reference to link the post to.
	 */
	public LocalPost addLocalPost(Object message, Object fileref);
	
	/**
	 * Add a new file from this node.  A post will be created
	 * for it.  The attached message will be added to the post.
	 * @param f - The file to add.
	 * @param message - The message File to link the file to.
	 */
	public LocalFileReference addLocalFile(File f, Object message);
	
	/**
	 * A friend wishes to join the network.  This is how you
	 * would sign his peer data so that he can join.
	 * @param peerdata - The peer data to sign.
	 * @return - The signed peer data that can be given back to your friend.
	 */
	public Peer signPeer(Peer peerdata);
	
	/**
	 * Set the network location for this node.  Most IP's will be DHCP
	 * so this is used to update the IP if needed.
	 * @param location - The updated network location.
	 */
	public void setMyLocation(Object location);
	
	/**
	 * Set the nickname
	 * @param location - The updated network location.
	 */
	public void setMyNickname(String nn);
	
	/**
	 * Set the network location for this node.  Most IP's will be DHCP
	 * so this is used to update the IP if needed.
	 * @param location - The updated network location.
	 */
	public void setMyIntroduction(String intro);
	
	/**
	 * Generate keys for this node.  This should only be done on the first
	 * system startup.  Doing this again will assign the node a new ID and
	 * it's peer data would need to be signed again to join the network.
	 */
	public void generateNewKeys();

	/**
	 * Return this node's peer data.
	 * @return
	 */
	public Peer getMyPeerData();
	
	/**
	 * Get a list of all valid peers that this node knows about.
	 * @return
	 */
	public List<Peer> getPeerList();

	/**
	 * Return the posts that were made from a given peer in pages.
	 * @param p - The peer to retrieve posts for.
	 * @param page - The page to return.
	 * @param pagesize - The page size.
	 * @return The list (page) of posts made by that peer as specified.
	 */
	public List<LocalPost> getPeerPostsByPage(Peer p, int page, int pagesize);

	/**
	 * Return posts from a peer by the sequence numbers.
	 * @param p - The peer to retrieve posts for.
	 * @param start - The starting sequence number.
	 * @param end - The ending sequence number.
	 * @return The list of posts made by that peer within this sequence number rage.
	 */
	public List<LocalPost> getPeerPostsByNumber(Peer p, long start, long end);

	/**
	 * Get posts linked to a given file.  
	 * @param dig The digest of the file to get posts for.
	 * @param page The page number of the posts to return
	 * @param pagesize The page size.
	 * @return The page of posts linked to the given file.
	 */
	public List<LocalPost> getFilePosts(Object dig, int page, int pagesize);
	
	/**
	 * Get all posts received after a given date/time.  This can 
	 * be used to only get new posts for the GUI.
	 * @param fromdate The date to return posts after.
	 * @return The list of posts received after the specified date/time.
	 */
	public List<LocalPost> getPosts(Date fromdate);
	
	/**
	 * Get a file reference for the given file digest.  The File
	 * in the file reference will be non-null if we have a local copy.
	 * @param dig - The digest of the file to download.
	 * @return The file reference.
	 */
	public LocalFileReference getFileReference(Object dig);
	
	/**
	 * Get file references by page. 
	 * @param page Page to return
	 * @param pagesize Page size
	 * @return The page specified. 
	 */
	public List<LocalFileReference> getFileReferences(int page, int pagesize);

	/**
	 * Get file references received after the given date.
	 * @param fromdate The date to return file references after.
	 * @return the list of file references after the given date.
	 */
	public List<LocalFileReference> getFileReferences(Date fromdate);
	
	/**
	 * Close all currently open connections.
	 */
	public void CloseConnections();
	
	/**
	 * Mark a peer as a bad peer.
	 * @param id The id of the peer that's bad.
	 */
	public void setBadPeer(Peer id);

}
