package org.ourfilesystem.security;

/*
OurFileSystem is a peer2peer file sharing program.
Copyright (C) 2012  Robert Gass

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/

import java.io.File;
import java.io.IOException;

import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.params.RSAKeyParameters;
import org.bouncycastle.crypto.params.RSAPrivateCrtKeyParameters;
import org.ourfilesystem.db.Peer;
import org.ourfilesystem.db.Post;
import org.ourfilesystem.utilities.BBytes;

public class CryptoDataBaseImpl implements CryptoDataBaseInterface {

	@Override
	public KeySet generateNewKeys() {
		AsymmetricCipherKeyPair encpair = SecurityTools.generateKeyPair();
		AsymmetricCipherKeyPair signpair = SecurityTools.generateKeyPair();
		KeySet ks = new KeySet();
		ks.setPrivateEncryptionKey(encpair.getPrivate());
		ks.setPrivateSigningKey(signpair.getPrivate());
		PublicKeySet pub = new PublicKeySet();
		pub.setPublicEncryptionKey(encpair.getPublic());
		pub.setPublicSigningKey(signpair.getPublic());
		ks.setPublicKeySet(pub);
		return ks;
	}

	@Override
	public boolean verifyPeer(PublicKeySetSigned pubsign, Object pub) {
		return SecurityTools.verifyPublicKeySetSigned(pubsign, (RSAKeyParameters)pub);
	}

	@Override
	public boolean digEquals(Object id0, Object id1) {
		if ( !( (id0 instanceof BBytes) && (id1 instanceof BBytes) ) ) {
			throw new RuntimeException("Expecting digests to be of type BBytes: " + id0 + ", " + id1);
		}
		return id0.equals(id1);
	}

	@Override
	public Peer signPeer(Peer p, Object peerid, Object privkey) {
		PublicKeySetSigned sig = SecurityTools.signPublicKeySet(p.getPeerKeysAndIdentity(), (BBytes)peerid, (RSAPrivateCrtKeyParameters) privkey);
		p.setPeerKeysAndIdentity(sig);
		return p;
	}
	
	@Override
	public boolean verifyPost(Post p, Object pubkey) {
		try {
			return SecurityTools.verifyPost(p, (RSAKeyParameters) pubkey);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return false;
	}

	@Override
	public Post signPost(Post p, Object peerid, Object privkey) {
		try {
			SecurityTools.signPost(p, (BBytes)peerid, (RSAPrivateCrtKeyParameters) privkey);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return p;
	}

	@Override
	public Object digestFile(File f) {
		try {
			return SecurityTools.digestFile(f);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}

	@Override
	public void selfSignMyPeerId(Peer p, Object privkey) {
		PublicKeySetSigned sig = SecurityTools.signPublicKeySet(p.getPeerKeysAndIdentity(), null, (RSAPrivateCrtKeyParameters) privkey);
		sig.getSignature().setPeerIdentifier(sig.getSignature().getDigest());
		p.setPeerKeysAndIdentity(sig);
	}

	@Override
	public void signLocation(Peer p, Object privkey) {
		BBytes dig = SecurityTools.digestPeerLocation(p);
		SignedDigest sg = SecurityTools.signDigest(dig.getBytes(), null, (RSAPrivateCrtKeyParameters) privkey);
		p.setLocationSignature(sg);
	}

	@Override
	public boolean verifyLocation(Peer p, Object pub) {
		BBytes dig = SecurityTools.digestPeerLocation(p);
		return SecurityTools.verifySignedDigest(p.getLocationSignature(), dig.getBytes(), (RSAKeyParameters) pub);
	}

}
