﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.IO.IsolatedStorage;
using System.Linq;
using System.Xml;

namespace I2PTalk
{
    public enum AuthorizationPolicy
    {
        AcceptAll,
        Ask,
        BlockAll
    }

    public enum IncomingChatMessageAction
    {
        BlinkTrayIcon,
        ShowBalloon,
        ShowChatWindow
    }

    public class Settings
    {
        readonly string settingsFileName = "settings.xml";

        public Identity Identity { get; set; }

        public bool ConnectAtStartup { get; set; }
        public AuthorizationPolicy AuthorizationPolicy { get; set; }
        public TimeSpan StatusUpdateInterval { get; set; }
        public TimeSpan StatusUpdateTimeout { get; set; }

        public bool SaveWindowLocation { get; set; }
        public bool ShowTimeStamps { get; set; }
        public Point WindowLocation { get; set; }
        public IncomingChatMessageAction IncomingChatMessageAction { get; set; }

        public string SamHost { get; set; }
        public int SamPort { get; set; }
        public int SamUdpPort { get; set; }
        public Dictionary<string, string> SamOptions { get; set; }        

        public Settings()
        {
            ResetToDefault();
        }

        public bool Load()
        {
            return Load(settingsFileName);
        }

        public bool Load(string fileName)
        {
            try
            {
                //using (IsolatedStorageFile isFile = IsolatedStorageFile.GetUserStoreForApplication())
                using (IsolatedStorageFileStream isFileStream = new IsolatedStorageFileStream(settingsFileName, FileMode.Open, FileAccess.Read))
                {
                    XmlDocument xmlDocument = new XmlDocument();

                    xmlDocument.Load(isFileStream);

                    if (xmlDocument.DocumentElement.Name != "I2PTalkSettings" || xmlDocument.DocumentElement.GetAttribute("Version") != "1.0")
                        return false;

                    XmlElement settingsElement = xmlDocument["I2PTalkSettings"];

                    Identity = Identity.FromXml(settingsElement["Identity"]);

                    ConnectAtStartup = Convert.ToBoolean(settingsElement["ConnectAtStartup"].InnerText);
                    AuthorizationPolicy = (AuthorizationPolicy)Enum.Parse(typeof(AuthorizationPolicy), settingsElement["AuthorizationPolicy"].InnerText);
                    StatusUpdateInterval = TimeSpan.FromMilliseconds(Convert.ToInt32(settingsElement["StatusUpdateInterval"].InnerText));
                    StatusUpdateTimeout = TimeSpan.FromMilliseconds(Convert.ToInt32(settingsElement["StatusUpdateTimeout"].InnerText));

                    SaveWindowLocation = Convert.ToBoolean(settingsElement["SaveWindowLocation"].InnerText);
                    string windowLocation = settingsElement["WindowLocation"].InnerText;
                    string[] wl = windowLocation.Split(new char[] { ',' }, 2);
                    WindowLocation = new Point(Convert.ToInt32(wl[0]), Convert.ToInt32(wl[1]));
                    ShowTimeStamps = Convert.ToBoolean(settingsElement["ShowTimeStamps"].InnerText);
                    IncomingChatMessageAction = (IncomingChatMessageAction)Enum.Parse(typeof(IncomingChatMessageAction), settingsElement["IncomingChatMessageAction"].InnerText);

                    SamHost = settingsElement["SamHost"].InnerText;
                    SamPort = Convert.ToInt32(settingsElement["SamPort"].InnerText);
                    SamUdpPort = Convert.ToInt32(settingsElement["SamUdpPort"].InnerText);
                    SamOptions = GetSamOptionsFromString(settingsElement["SamOptions"].InnerText);
                }

                return true;
            }
            catch
            {
                return false;
            }
        }

        private string GetSamOptionsAsString()
        {
            return string.Join(";", SamOptions.Select(option => option.Key + "=" + option.Value));
        }

        private Dictionary<string, string> GetSamOptionsFromString(string s)
        {
            Dictionary<string, string> options = new Dictionary<string, string>();

            foreach (string part in s.Split(';'))
            {
                if (part == string.Empty)
                    continue;

                string[] keyvalue = part.Split('=');

                if (keyvalue.Length != 2)
                    return new Dictionary<string,string>();

                options.Add(keyvalue[0], keyvalue[1]);
            }

            return options;
        }

        public bool Save()
        {
            return Save(settingsFileName);
        }

        public bool Save(string fileName)
        {
            try
            {
                //using (IsolatedStorageFile isFile = IsolatedStorageFile.GetMachineStoreForApplication())
                using (IsolatedStorageFileStream isFileStream = new IsolatedStorageFileStream(settingsFileName, FileMode.Create, FileAccess.Write))
                using (XmlWriter xmlWriter = XmlWriter.Create(isFileStream))
                {
                    xmlWriter.WriteStartDocument();
                    xmlWriter.WriteStartElement("I2PTalkSettings");
                    xmlWriter.WriteAttributeString("Version", "1.0");

                    Identity.WriteToXml(xmlWriter);

                    xmlWriter.WriteElementBoolean("ConnectAtStartup", ConnectAtStartup);
                    xmlWriter.WriteElementString("AuthorizationPolicy", Enum.GetName(typeof(AuthorizationPolicy), AuthorizationPolicy));
                    xmlWriter.WriteElementInt("StatusUpdateInterval", (int)StatusUpdateInterval.TotalMilliseconds);
                    xmlWriter.WriteElementInt("StatusUpdateTimeout", (int)StatusUpdateTimeout.TotalMilliseconds);

                    xmlWriter.WriteElementBoolean("SaveWindowLocation", SaveWindowLocation);
                    xmlWriter.WriteElementString("WindowLocation", string.Format("{0},{1}", WindowLocation.X, WindowLocation.Y));
                    xmlWriter.WriteElementBoolean("ShowTimeStamps", ShowTimeStamps);
                    xmlWriter.WriteElementString("IncomingChatMessageAction", Enum.GetName(typeof(IncomingChatMessageAction), IncomingChatMessageAction));

                    xmlWriter.WriteElementString("SamHost", SamHost);
                    xmlWriter.WriteElementInt("SamPort", SamPort);
                    xmlWriter.WriteElementInt("SamUdpPort", SamUdpPort);
                    xmlWriter.WriteElementString("SamOptions", GetSamOptionsAsString());

                    xmlWriter.WriteEndElement();
                    xmlWriter.WriteEndDocument();
                }

                return true;
            }
            catch
            {
                return false;
            }
        }

        public void ResetToDefault()
        {
            Identity = null;

            ConnectAtStartup = false;
            AuthorizationPolicy = AuthorizationPolicy.Ask;
            StatusUpdateInterval = TimeSpan.FromSeconds(20);
            StatusUpdateTimeout = TimeSpan.FromSeconds(75);

            SaveWindowLocation = true;
            WindowLocation = Point.Empty;
            ShowTimeStamps = true;
            IncomingChatMessageAction = IncomingChatMessageAction.ShowChatWindow;

            SamHost = "localhost";
            SamPort = 7656;
            SamUdpPort = 7655;
            SamOptions = new Dictionary<string, string>();
            SamOptions.Add("inbound.length", "3");
            SamOptions.Add("outbound.length", "3");
            SamOptions.Add("inbound.lengthVariance", "0");
            SamOptions.Add("outbound.lengthVariance", "0");
            SamOptions.Add("inbound.quantity", "2");
            SamOptions.Add("outbound.quantity", "2");
            SamOptions.Add("inbound.backupQuantity", "0");
            SamOptions.Add("outbound.backupQuantity", "0");
        }
    }
}
