﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Net.Sockets;
using System.Runtime.Serialization;
using System.Text;
using System.Threading.Tasks;

namespace I2PTalk.SamLib
{
    class SamReply
    {
        public string Description { get; private set; }
        public IReadOnlyDictionary<string, string> Properties { get; private set; }

        private SamReply()
        {
        }

        public static SamReply Parse(string s)
        {
            List<string> tokens = new List<string>();

            int i = 0;
            int tokenStart = 0;

            while (i < s.Length)
            {
                char c = s[i];

                if (c == ' ')
                {
                    if (tokenStart < i - 1)
                        tokens.Add(s.Substring(tokenStart, i - tokenStart));

                    tokenStart = i + 1;
                }
                else if (c == '"')
                {
                    do
                        i++;
                    while (i < s.Length && s[i] != '"');

                    if (i == s.Length)
                        return null;
                }

                i++;
            }

            if (tokenStart < s.Length)
                tokens.Add(s.Substring(tokenStart));

            int numDesc = 0;

            while (numDesc < tokens.Count && !tokens[numDesc].Contains("="))
                numDesc++;

            if (numDesc == 0)
                return null;

            Dictionary<string, string> properties = new Dictionary<string, string>(tokens.Count - numDesc);

            for (int j = numDesc; j < tokens.Count; j++)
            {
                string[] propValue = tokens[j].Split(new char[] { '=' }, 2);

                if (propValue.Length != 2)
                    return null;

                properties.Add(propValue[0], propValue[1].Trim('"'));
            }

            SamReply reply = new SamReply();

            reply.Description = string.Join(" ", tokens.ToArray(), 0, numDesc);
            reply.Properties = properties;

            return reply;
        }

        public bool Success
        {
            get
            {
                string result;

                return Properties.TryGetValue("RESULT", out result) && result == "OK";
            }
        }
    }

    [Serializable]
    public class SamException : Exception
    {
        public string SamError { get; private set; }
        public string SamMessage { get; private set; }

        public SamException()
        {
        }

        public SamException(string message)
            : base(message)
        {
        }

        protected SamException(SerializationInfo info, StreamingContext context)
            : base(info, context)
        {
            SamError = info.GetString("SamError");
            SamMessage = info.GetString("SamMessage");
        }

        internal static SamException FromReply(SamReply reply)
        {
            string samError;

            if (!reply.Properties.TryGetValue("RESULT", out samError))
                samError = null;

            string samMessage;

            if (!reply.Properties.TryGetValue("MESSAGE", out samMessage))
                samMessage = null;

            string message = "The SAM bridge returned an error.";

            if (!string.IsNullOrEmpty(samError))
                message += string.Format(" Error: {0}", samError);
            if (!string.IsNullOrEmpty(samMessage))
                message += string.Format(" Message: {0}", samMessage);

            return new SamException(message) { SamError = samError, SamMessage = samMessage };
        }

        public override void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            base.GetObjectData(info, context);

            info.AddValue("SamError", SamError);
            info.AddValue("SamMessage", SamMessage);
        }
    }

    public enum SessionStyle
    {
        Stream,
        Datagram,
        Raw
    }

    public class KeyPair
    {
        public string PublicKey { get; private set; }
        public string PrivateKey { get; private set; }

        internal KeyPair(string publicKey, string privateKey)
        {
            PublicKey = publicKey;
            PrivateKey = privateKey;
        }
    }
    
    public class SamClient
    {
        internal TcpClient TcpClient { get; private set; }
        NetworkStream networkStream;

        public string Host { get; private set; }
        public int Port { get; private set; }
        
        public SamClient()
        {
        }

        private async Task SendCommandAsync(string s)
        {
            byte[] buffer = Encoding.ASCII.GetBytes(s + "\n");

            await networkStream.WriteAsync(buffer, 0, buffer.Length);
        }

        internal async Task<string> ReceiveLineAsync()
        {
            StringBuilder line = new StringBuilder();

            byte[] buffer = new byte[1];

            while (await networkStream.ReadAsync(buffer, 0, buffer.Length) == 1 && buffer[0] != '\n')
                line.Append((char)buffer[0]);

            return line.ToString();
        }

        internal async Task<SamReply> ReceiveReplyAsync()
        {
            return SamReply.Parse(await ReceiveLineAsync());

            //using (StreamReader streamReader = new StreamReader(networkStream, Encoding.ASCII, false, 256, true))
            //    return Reply.Parse(await streamReader.ReadLineAsync() ?? string.Empty);
        }

        public async Task ConnectAsync(string host, int port)
        {
            this.Host = host;
            this.Port = port;

            TcpClient = new TcpClient();

            await TcpClient.ConnectAsync(host, port);

            networkStream = TcpClient.GetStream();

            try
            {
                await PerformHandshakeAsync();
            }
            catch (SamException)
            {
                TcpClient.Close();
                TcpClient = null;
                throw;
            }  
        }

        public void Disconnect()
        {
            if (TcpClient != null)
                TcpClient.Close();

            TcpClient = null;
        }

        private async Task PerformHandshakeAsync()
        {
            await SendCommandAsync("HELLO VERSION MIN=3.0 MAX=3.0");

            SamReply reply = await ReceiveReplyAsync();

            if (reply == null || reply.Description != "HELLO REPLY")
                throw new SamException("Invalid SAM reply.");

            if (!reply.Success)
                throw SamException.FromReply(reply);
        }

        internal async Task<string> CreateSessionAsync(SessionStyle style, string nickname, string destination = null, Dictionary<string, string> options = null)
        {
            StringBuilder request = new StringBuilder();

            request.Append("SESSION CREATE");

            switch (style)
            {
                case SessionStyle.Stream:
                    request.Append(" STYLE=STREAM");
                    break;
                case SessionStyle.Datagram:
                    request.Append(" STYLE=DATAGRAM");
                    break;
                case SessionStyle.Raw:
                    request.Append(" STYLE=RAW");
                    break;
            }

            request.Append(" ID=" + nickname);

            if (string.IsNullOrEmpty(destination))
                request.Append(" DESTINATION=TRANSIENT");
            else
            {
                request.Append(" DESTINATION=");
                request.Append(destination);
            }

            if (options != null)
                foreach (KeyValuePair<string, string> option in options)
                    request.Append(' ' + option.Key + '=' + option.Value);

            await SendCommandAsync(request.ToString());

            SamReply reply = await ReceiveReplyAsync();

            if (reply == null || reply.Description != "SESSION STATUS")
                throw new SamException("Invalid SAM reply.");

            if (reply.Success)
                return reply.Properties["DESTINATION"];
            else
                throw SamException.FromReply(reply);
        }

        internal async Task StreamConnectAsync(string nickname, string destination)
        {
            await SendCommandAsync(string.Format("STREAM CONNECT ID={0} DESTINATION={1}", nickname, destination));

            SamReply reply = await ReceiveReplyAsync();

            if (reply == null || reply.Description != "STREAM STATUS")
                throw new SamException("Invalid SAM reply.");

            if (!reply.Success)
                throw SamException.FromReply(reply);
        }

        internal async Task StreamAcceptAsync(string nickname)
        {
            await SendCommandAsync("STREAM ACCEPT ID=" + nickname);

            SamReply reply = await ReceiveReplyAsync();

            if (reply == null || reply.Description != "STREAM STATUS")
                throw new SamException("Invalid SAM reply.");

            if (!reply.Success)
                throw SamException.FromReply(reply);
        }

        public async Task<string> NamingLookupAsync(string name)
        {
            await SendCommandAsync("NAMING LOOKUP NAME=" + name);

            SamReply reply = await ReceiveReplyAsync();

            if (reply == null || reply.Description != "NAMING REPLY")
                throw new SamException("Invalid SAM reply.");

            if (reply.Success)
                return reply.Properties["VALUE"];
            else
                throw SamException.FromReply(reply);
        }

        public async Task<KeyPair> GenerateKeyPairAsync()
        {
            await SendCommandAsync("DEST GENERATE");

            SamReply reply = await ReceiveReplyAsync();

            if (reply == null || reply.Description != "DEST REPLY")
                throw new SamException("Invalid SAM reply.");

            return new KeyPair(reply.Properties["PUB"], reply.Properties["PRIV"]);
        }

        public bool Connected
        {
            get
            {
                return TcpClient != null && TcpClient.Connected;
            }
        }
    }
}