﻿using System;
using System.Collections.Generic;
using System.Net.Sockets;
using System.Text;
using System.Threading.Tasks;

namespace I2PTalk.SamLib
{
    public class DatagramSession
    {
        SamClient samClient = new SamClient();
        UdpClient udpClient;

        public string Nickname { get; private set; }
        public string Destination { get; private set; }
        Dictionary<string, string> options;

        bool rawDatagrams;

        public DatagramSession(string nickname, bool rawDatagrams, string destination = null, Dictionary<string, string> options = null)
        {
            Nickname = nickname;
            Destination = destination;
            this.options = options;
            this.rawDatagrams = rawDatagrams;

            if (this.options == null)
                this.options = new Dictionary<string, string>(1);

            if (!this.options.ContainsKey("inbound.nickname"))
                this.options.Add("inbound.nickname", Nickname);
        }

        public async Task OpenAsync(string samHost, int samPort, int samUdpPort)
        {
            await samClient.ConnectAsync(samHost, samPort);

            Destination = await samClient.CreateSessionAsync(rawDatagrams ? SessionStyle.Raw : SessionStyle.Datagram, Nickname, Destination, options);

            udpClient = new UdpClient(samHost, samUdpPort);
        }

        public void Close()
        {
            samClient.Disconnect();

            if (udpClient != null)
                udpClient.Close();
        }

        public async Task<int> SendAsync(string destination, byte[] data, int index, int length)
        {
            string header = "3.0 " + Nickname + " " + destination + '\n';

            byte[] datagram = new byte[header.Length + length];

            Encoding.ASCII.GetBytes(header, 0, header.Length, datagram, 0);

            Array.Copy(data, index, datagram, header.Length, length);

            return await udpClient.SendAsync(datagram, datagram.Length);
        }

        public async Task<ReceivedDatagram> ReceiveAsync()
        {
            SamReply reply = await samClient.ReceiveReplyAsync();

            if ((!rawDatagrams && reply.Description != "DATAGRAM RECEIVED") ||
                (rawDatagrams && reply.Description != "RAW RECEIVED"))
                throw new SamException("Invalid SAM reply.");

            string destination = null;

            if (!rawDatagrams)
                destination = reply.Properties["DESTINATION"];

            int size = Convert.ToInt32(reply.Properties["SIZE"]);

            if (size < 0 || size > 32768)
                throw new SamException("Invalid SAM reply.");

            byte[] data = new byte[size];

            await samClient.TcpClient.GetStream().ReadAsync(data, 0, data.Length);

            return new ReceivedDatagram(destination, data);
        }

        public bool Alive
        {
            get
            {
                return samClient.TcpClient.Connected;
            }
        }
    }

    public class ReceivedDatagram
    {
        public string Destination { get; private set; }
        public byte[] Data { get; private set; }

        internal ReceivedDatagram(string destination, byte[] data)
        {
            Destination = destination;
            Data = data;
        }
    }
}
