﻿using System;
using System.Collections.Generic;
using System.Xml;

namespace I2PTalk
{
    public enum ContactStatus
    {
        Offline,
        Online,
        Away,
        Busy,
        Invisible
    }

    public class Contact
    {
        public string Name { get; set; }
        public string DisplayName { get; set; }
        public Destination Destination { get; set; }
        public string Profile { get; set; }
        public ContactStatus Status { get; set; }
        public DateTime LastHeard { get; set; }
        public DateTime? LastStatusUpdateSent { get; set; }
        public bool PendingAuthorizationRequest { get; set; }
        public string AuthorizationMessage { get; set; }

        public Contact()
        {
        }

        public static Contact CreateNew(string name, string displayName, Destination destination, string profile, bool pendingAuthorizationRequest, string authorizationMessage)
        {
            Contact contact = new Contact();

            contact.Name = name;
            contact.DisplayName = displayName;
            contact.Destination = destination;
            contact.Profile = profile;
            contact.Status = ContactStatus.Offline;
            contact.PendingAuthorizationRequest = pendingAuthorizationRequest;
            contact.AuthorizationMessage = authorizationMessage;

            return contact;
        }

        public string GetI2PTalkAddress()
        {
            return "i2ptalk://" + Name + '/' + Destination.Base32.Substring(0, 52);
        }

        public static Contact FromXml(XmlElement element)
        {
            Contact contact = new Contact();

            contact.Name = element["Name"].InnerText;
            contact.DisplayName = element["DisplayName"].InnerText;
            string dest = element["Destination"].InnerText;
            contact.Destination = Destination.Parse(element["Destination"].InnerText);
            contact.Profile = element["Profile"].InnerText;
            contact.PendingAuthorizationRequest = Convert.ToBoolean(element["PendingAuthorizationRequest"].InnerText);
            if (element["AuthorizationMessage"] != null)
                contact.AuthorizationMessage = element["AuthorizationMessage"].InnerText;

            return contact;
        }

        public void WriteToXml(XmlWriter xmlWriter)
        {
            xmlWriter.WriteStartElement("Contact");
            xmlWriter.WriteElementString("Name", Name);
            xmlWriter.WriteElementString("DisplayName", DisplayName);
            xmlWriter.WriteElementString("Destination", Destination.ToString());
            xmlWriter.WriteElementString("Profile", Profile);
            xmlWriter.WriteElementBoolean("PendingAuthorizationRequest", PendingAuthorizationRequest);
            if (AuthorizationMessage != null)
                xmlWriter.WriteElementString("AuthorizationMessage", AuthorizationMessage);
            xmlWriter.WriteEndElement();
        }
    }

    public class BlocklistEntry
    {
        public string ContactName { get; private set; }
        public string ContactDisplayName { get; private set; }
        public Destination ContactDestination { get; private set; }

        public BlocklistEntry(string contactName, string contactDisplayName, Destination contactDestination)
        {
            ContactName = contactName;
            ContactDisplayName = contactDisplayName;
            ContactDestination = contactDestination;
        }

        public string GetI2PTalkAddress()
        {
            return "i2ptalk://" + ContactName + '/' + ContactDestination.Base32.Substring(0, 52);
        }

        public static BlocklistEntry FromXml(XmlElement element)
        {
            string contactName = element["ContactName"].InnerText;
            string contactDisplayName = element["ContactDisplayName"].InnerText;
            Destination contactDestination = Destination.Parse(element["ContactDestination"].InnerText);

            return new BlocklistEntry(contactName, contactDisplayName, contactDestination);
        }

        public void WriteToXml(XmlWriter xmlWriter)
        {
            xmlWriter.WriteStartElement("BlocklistEntry");
            xmlWriter.WriteElementString("ContactName", ContactName);
            xmlWriter.WriteElementString("ContactDisplayName", ContactDisplayName);
            xmlWriter.WriteElementString("ContactDestination", ContactDestination.ToString());           
            xmlWriter.WriteEndElement();
        }
    }

    public class AuthorizationRequest
    {
        public string ContactName { get; private set; }
        public string ContactDestination { get; private set; }
        public string ContactProfile { get; private set; }
        public string Message { get; private set; }

        public AuthorizationRequest(string contactName, string contactDestination, string contactProfile, string message)
        {
            ContactName = contactName;
            ContactDestination = contactDestination;
            ContactProfile = contactProfile;
            Message = message;
        }

        public static AuthorizationRequest FromXml(XmlElement element)
        {
            string contactName = element["ContactName"].InnerText;
            string contactDestination = element["ContactDestination"].InnerText;
            string contactProfile = element["ContactProfile"].InnerText;
            string message = element["Message"].InnerText;

            return new AuthorizationRequest(contactName, contactDestination, contactProfile, message);
        }

        public void WriteToXml(XmlWriter xmlWriter)
        {
            xmlWriter.WriteStartElement("AuthorizationRequest");
            xmlWriter.WriteElementString("ContactName", ContactName);           
            xmlWriter.WriteElementString("ContactDestination", ContactDestination);
            xmlWriter.WriteElementString("ContactProfile", ContactProfile);
            xmlWriter.WriteElementString("Message", Message);
            xmlWriter.WriteEndElement();
        }
    }

    public class Identity
    {
        public string Name { get; private set; }
        public string Destination { get; set; }
        public string Profile { get; set; }
        public List<Contact> Contacts { get; private set; }
        public List<BlocklistEntry> Blocklist { get; private set; }
        public List<AuthorizationRequest> AuthorizationRequests { get; private set; }

        private Identity()
        {
        }

        public static Identity CreateNew(string name, string profile)
        {
            Identity identity = new Identity();

            identity.Name = name;
            identity.Profile = profile;
            identity.Contacts = new List<Contact>();
            identity.Blocklist = new List<BlocklistEntry>();
            identity.AuthorizationRequests = new List<AuthorizationRequest>();

            return identity;
        }

        public string GetI2PTalkAddress()
        {
            if (Destination != null)
                return "i2ptalk://" + Name + '/' + I2PTalk.Destination.FromBase64(Destination.Substring(0, 516)).Base32.Substring(0, 52);
            else
                return string.Empty;
        }

        public static Identity FromXml(XmlElement element)
        {
            Identity identity = new Identity();

            identity.Name = element["Name"].InnerText;
            if (element["Destination"] != null)
                identity.Destination = element["Destination"].InnerText;
            identity.Profile = element["Profile"].InnerText;

            identity.Contacts = new List<Contact>();
            foreach (XmlElement e in element["Contacts"].ChildNodes)
                identity.Contacts.Add(Contact.FromXml(e));

            identity.Blocklist = new List<BlocklistEntry>();
            foreach (XmlElement e in element["Blocklist"].ChildNodes)
                identity.Blocklist.Add(BlocklistEntry.FromXml(e));

            identity.AuthorizationRequests = new List<AuthorizationRequest>();
            foreach (XmlElement e in element["AuthorizationRequests"].ChildNodes)
                identity.AuthorizationRequests.Add(AuthorizationRequest.FromXml(e));

            return identity;
        }

        public void WriteToXml(XmlWriter xmlWriter)
        {
            xmlWriter.WriteStartElement("Identity");
            xmlWriter.WriteElementString("Name", Name);
            if (Destination != null)
                xmlWriter.WriteElementString("Destination", Destination);
            xmlWriter.WriteElementString("Profile", Profile);

            xmlWriter.WriteStartElement("Contacts");
            foreach (Contact contact in Contacts)
                contact.WriteToXml(xmlWriter);
            xmlWriter.WriteEndElement();

            xmlWriter.WriteStartElement("Blocklist");
            foreach (BlocklistEntry blocklistEntry in Blocklist)
                blocklistEntry.WriteToXml(xmlWriter);
            xmlWriter.WriteEndElement();

            xmlWriter.WriteStartElement("AuthorizationRequests");
            foreach (AuthorizationRequest authorizationRequest in AuthorizationRequests)
                authorizationRequest.WriteToXml(xmlWriter);
            xmlWriter.WriteEndElement();

            xmlWriter.WriteEndElement();
        }
    }
}
