﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace I2PTalk.ChatBox
{
    class Span
    {
        public int Start { get; private set; }
        public int Length { get; private set; }

        public Span(int start, int length)
        {
            Start = start;
            Length = length;
        }
    }

    public abstract class ChatBoxItem
    {
        internal int TextPosition { get; set; }
        internal abstract string GetText();

        internal virtual void Format(RichTextBox richTextBox)
        {
        }

        internal virtual IEnumerable<Span> GetHyperlinks()
        {
            return Enumerable.Empty<Span>();
        }

        internal virtual void OnLinkClicked(Span link)
        {
        }
    }

    public class MessageChatBoxItem : ChatBoxItem
    {
        public DateTimeOffset Date { get; private set; }
        public string Nickname { get; private set; }
        public string Message { get; private set; }

        public MessageChatBoxItem(DateTimeOffset date, string nickname, string message)
        {
            Date = date;
            Nickname = nickname;
            Message = message;
        }

        internal override string GetText()
        {
            TimeSpan roundedTime = TimeSpan.FromSeconds((int)Date.TimeOfDay.TotalSeconds);

            return string.Format("({0})  {1}: {2}\n", roundedTime, Nickname, Message);
        }

        internal override void Format(RichTextBox richTextBox)
        {
            richTextBox.Select(TextPosition + 12, Nickname.Length + 1);
            richTextBox.SelectionFont = new Font(richTextBox.SelectionFont, FontStyle.Bold);

            foreach (Span span in GetHyperlinks())
            {
                richTextBox.Select(TextPosition + span.Start, span.Length);
                richTextBox.SelectionFont = new Font(richTextBox.SelectionFont, FontStyle.Underline);
                richTextBox.SelectionColor = Color.Blue;
            }

            richTextBox.DeselectAll();
        }

        internal override IEnumerable<Span> GetHyperlinks()
        {
            string[] protocols = { "http://", "https://", "ftp://", "nntp://", "mailto:", "ssh://", "scp://", "sftp://", "irc://", "i2ptalk://" };

            List<Span> hyperlinks = new List<Span>();

            string[] tokens = Message.Split(new char[] { ' ', '(', ')', '[', ']', '{', '}', ',', ';', '\t', '"' }, StringSplitOptions.None);

            int start = 0;

            for (int i = 0; i < tokens.Length; i++)
            {
                bool isLink = false;

                string token = tokens[i].TrimEnd('.', '?', '!');

                foreach (string protocol in protocols)
                    if (token.StartsWith(protocol, StringComparison.InvariantCultureIgnoreCase))
                    {
                        isLink = true;
                        break;
                    }

                if (token.TrimEnd('/').EndsWith(".i2p", StringComparison.InvariantCulture) ||
                    token.TrimEnd('/').EndsWith(".onion", StringComparison.InvariantCulture))
                    isLink = true;

                if (isLink)
                    hyperlinks.Add(new Span(14 + Nickname.Length + start, token.Length));

                start += tokens[i].Length + 1;
            }

            return hyperlinks;
        }

        internal override void OnLinkClicked(Span link)
        {
            string linkText = GetText().Substring(link.Start, link.Length);

            if (linkText.StartsWith("i2ptalk://", StringComparison.InvariantCultureIgnoreCase))
                Core.Instance.AddContact(linkText);
            else
            {
                try
                {
                    Process.Start(linkText);
                }
                catch
                {
                    MessageBox.Show("Could not open link.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
        }
    }
}
