﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Net.Sockets;
using System.Runtime.Serialization;
using System.Text;
using System.Threading.Tasks;

namespace SamLib
{
    internal class Reply
    {
        public string Description { get; private set; }
        public IReadOnlyDictionary<string, string> Properties { get; private set; }

        private Reply()
        {
        }

        public static Reply Parse(string s)
        {
            List<string> tokens = new List<string>();

            int i = 0;
            int tokenStart = 0;

            while (i < s.Length)
            {
                char c = s[i];

                if (c == ' ')
                {
                    if (tokenStart < i - 1)
                        tokens.Add(s.Substring(tokenStart, i - tokenStart));

                    tokenStart = i + 1;
                }
                else if (c == '"')
                {
                    do
                        i++;
                    while (i < s.Length && s[i] != '"');

                    if (i == s.Length)
                        return null;
                }

                i++;
            }

            if (tokenStart < s.Length)
                tokens.Add(s.Substring(tokenStart));

            int numDesc = 0;

            while (numDesc < tokens.Count && !tokens[numDesc].Contains("="))
                numDesc++;

            if (numDesc == 0)
                return null;

            Dictionary<string, string> properties = new Dictionary<string, string>(tokens.Count - numDesc);

            for (int j = numDesc; j < tokens.Count; j++)
            {
                string[] propValue = tokens[j].Split(new char[] { '=' }, 2);

                if (propValue.Length != 2)
                    return null;

                properties.Add(propValue[0], propValue[1].Trim('"'));
            }

            Reply reply = new Reply();

            reply.Description = string.Join(" ", tokens.ToArray(), 0, numDesc);
            reply.Properties = new ReadOnlyDictionary<string, string>(properties);

            return reply;
        }

        public bool Success
        {
            get
            {
                string result;

                return Properties.TryGetValue("RESULT", out result) && result == "OK";
            }
        }
    }

    [Serializable]
    public class SamException : Exception
    {
        public string SamError { get; private set; }
        public string SamMessage { get; private set; }

        public SamException()
        {
        }

        public SamException(string message) : base(message)
        {
        }

        protected SamException(SerializationInfo info, StreamingContext context)
            : base(info, context) 
        {
        }

        internal static SamException FromReply(Reply reply)
        {
            string samError;

            if (!reply.Properties.TryGetValue("RESULT", out samError))
                samError = null;

            string samMessage;

            if (!reply.Properties.TryGetValue("MESSAGE", out samMessage))
                samMessage = null;

            string message = "The SAM bridge returned an error.";

            if (!string.IsNullOrWhiteSpace(samError))
                message += string.Format(" Error: {0}", samError);
            if (!string.IsNullOrWhiteSpace(samMessage))
                message += string.Format(" Message: {0}", samMessage);
            
            return new SamException(message) { SamError = samError, SamMessage = samMessage };
        }
    }

    public enum SessionStyle
    {
        Stream,
        Datagram,
        Raw
    }

    public class KeyPair
    {
        public string PublicKey { get; private set; }
        public string PrivateKey { get; private set; }

        internal KeyPair(string publicKey, string privateKey)
        {
            PublicKey = publicKey;
            PrivateKey = privateKey;
        }
    }
    
    public class SamClient
    {
        internal TcpClient TcpClient { get; private set; }
        NetworkStream networkStream;

        public string Host { get; private set; }
        public int Port { get; private set; }
        
        public SamClient()
        {
        }

        private async Task SendCommand(string s)
        {
            byte[] buffer = Encoding.ASCII.GetBytes(s + "\n");

            await networkStream.WriteAsync(buffer, 0, buffer.Length);
        }

        internal async Task<string> ReceiveLine()
        {
            StringBuilder line = new StringBuilder();

            byte[] buffer = new byte[1];

            while (await networkStream.ReadAsync(buffer, 0, buffer.Length) == 1 && buffer[0] != '\n')
                line.Append((char)buffer[0]);

            return line.ToString();
        }

        internal async Task<Reply> ReceiveReply()
        {
            return Reply.Parse(await ReceiveLine());

            //using (StreamReader streamReader = new StreamReader(networkStream, Encoding.ASCII, false, 256, true))
            //    return Reply.Parse(await streamReader.ReadLineAsync() ?? string.Empty);
        }

        public async Task ConnectAsync(string host, int port)
        {
            this.Host = host;
            this.Port = port;

            TcpClient = new TcpClient();

            await TcpClient.ConnectAsync(host, port);

            networkStream = TcpClient.GetStream();

            try
            {
                await PerformHandshake();
            }
            catch (SamException)
            {
                TcpClient.Close();
                TcpClient = null;
                throw;
            }  
        }

        public void Disconnect()
        {
            if (TcpClient != null)
                TcpClient.Close();

            TcpClient = null;
        }

        private async Task PerformHandshake()
        {
            await SendCommand("HELLO VERSION MIN=3.0 MAX=3.0");

            Reply reply = await ReceiveReply();

            if (reply == null || reply.Description != "HELLO REPLY")
                throw new SamException("Invalid SAM reply.");

            if (!reply.Success)
                throw SamException.FromReply(reply);
        }

        internal async Task<string> CreateSession(SessionStyle style, string nickname, string destination = null, Dictionary<string, string> options = null)
        {
            StringBuilder request = new StringBuilder();

            request.Append("SESSION CREATE");

            switch (style)
            {
                case SessionStyle.Stream:
                    request.Append(" STYLE=STREAM");
                    break;
                case SessionStyle.Datagram:
                    request.Append(" STYLE=DATAGRAM");
                    break;
                case SessionStyle.Raw:
                    request.Append(" STYLE=RAW");
                    break;
            }

            request.Append(" ID=" + nickname);

            if (string.IsNullOrEmpty(destination))
                request.Append(" DESTINATION=TRANSIENT");
            else
            {
                request.Append(" DESTINATION=");
                request.Append(destination);
            }

            if (options != null)
                foreach (KeyValuePair<string, string> option in options)
                    request.Append(' ' + option.Key + '=' + option.Value);

            await SendCommand(request.ToString());

            Reply reply = await ReceiveReply();

            if (reply == null || reply.Description != "SESSION STATUS")
                throw new SamException("Invalid SAM reply.");

            if (reply.Success)
                return reply.Properties["DESTINATION"];
            else
                throw SamException.FromReply(reply);
        }

        internal async Task StreamConnect(string nickname, string destination)
        {
            await SendCommand(string.Format("STREAM CONNECT ID={0} DESTINATION={1}", nickname, destination));

            Reply reply = await ReceiveReply();

            if (reply == null || reply.Description != "STREAM STATUS")
                throw new SamException("Invalid SAM reply.");

            if (!reply.Success)
                throw SamException.FromReply(reply);
        }

        internal async Task StreamAccept(string nickname)
        {
            await SendCommand("STREAM ACCEPT ID=" + nickname);

            Reply reply = await ReceiveReply();

            if (reply == null || reply.Description != "STREAM STATUS")
                throw new SamException("Invalid SAM reply.");

            if (!reply.Success)
                throw SamException.FromReply(reply);
        }

        public async Task<string> NamingLookup(string name)
        {
            await SendCommand("NAMING LOOKUP NAME=" + name);

            Reply reply = await ReceiveReply();

            if (reply == null || reply.Description != "NAMING REPLY")
                throw new SamException("Invalid SAM reply.");

            if (reply.Success)
                return reply.Properties["VALUE"];
            else
                throw SamException.FromReply(reply);
        }

        public async Task<KeyPair> GenerateKeyPair()
        {
            await SendCommand("DEST GENERATE");

            Reply reply = await ReceiveReply();

            if (reply == null || reply.Description != "DEST REPLY")
                throw new Exception("Invalid SAM reply.");

            return new KeyPair(reply.Properties["PUB"], reply.Properties["PRIV"]);
        }

        public bool Connected
        {
            get
            {
                return TcpClient != null && TcpClient.Connected;
            }
        }
    }

    public class StreamSession
    {
        SamClient samClient = new SamClient();

        public string Nickname { get; private set; }
        public string Destination { get; private set; }
        Dictionary<string, string> options;

        public StreamSession(string nickname, string destination = null, Dictionary<string, string> options = null)
        {
            Nickname = nickname;
            Destination = destination;
            this.options = options;

            if (this.options == null)
                this.options = new Dictionary<string, string>(1);

            if (!this.options.ContainsKey("inbound.nickname"))
                this.options.Add("inbound.nickname", Nickname);
        }

        public async Task Open(string samHost, int samPort)
        {
            await samClient.ConnectAsync(samHost, samPort);

            Destination = await samClient.CreateSession(SessionStyle.Stream, Nickname, Destination, options);
        }

        public void Close()
        {
            samClient.Disconnect();
        }

        public async Task<TcpClient> Connect(string destination)
        {
            SamClient samClient2 = new SamClient();

            await samClient2.ConnectAsync(samClient.Host, samClient.Port);

            try
            {
                await samClient2.StreamConnect(Nickname, destination);
            }
            catch (SamException)
            {
                samClient2.Disconnect();
                throw;
            }

            return samClient2.TcpClient;
        }

        public async Task<AcceptedStream> Accept()
        {
            SamClient samClient2 = new SamClient();

            await samClient2.ConnectAsync(samClient.Host, samClient.Port);

            try
            {
                await samClient2.StreamAccept(Nickname);
            }
            catch (SamException)
            {
                samClient2.Disconnect();
                throw;
            }

            string clientDestination = await samClient2.ReceiveLine();

            return new AcceptedStream(clientDestination, samClient2.TcpClient);
        }

        public bool Alive
        {
            get
            {
                return samClient.Connected;
            }
        }
    }

    public class AcceptedStream
    {
        public string ClientDestination { get; private set; }
        public TcpClient TcpClient { get; private set; }

        internal AcceptedStream(string clientDestination, TcpClient tcpClient)
        {
            ClientDestination = clientDestination;
            TcpClient = tcpClient;
        }
    }

    public class DatagramSession
    {
        SamClient samClient = new SamClient();
        UdpClient udpClient;

        public string Nickname { get; private set; }
        public string Destination { get; private set; }
        Dictionary<string, string> options;

        bool rawDatagrams;

        public DatagramSession(string nickname, bool rawDatagrams, string destination = null, Dictionary<string, string> options = null)
        {
            Nickname = nickname;
            Destination = destination;
            this.options = options;
            this.rawDatagrams = rawDatagrams;

            if (this.options == null)
                this.options = new Dictionary<string, string>(1);

            if (!this.options.ContainsKey("inbound.nickname"))
                this.options.Add("inbound.nickname", Nickname);
        }

        public async Task Open(string samHost, int samPort, int samUdpPort)
        {
            await samClient.ConnectAsync(samHost, samPort);

            Destination = await samClient.CreateSession(rawDatagrams ? SessionStyle.Raw : SessionStyle.Datagram, Nickname, Destination, options);

            udpClient = new UdpClient(samHost, samUdpPort);
        }

        public void Close()
        {
            samClient.Disconnect();

            if (udpClient != null)
                udpClient.Close();
        }

        public async Task<int> Send(string destination, byte[] data)
        {
            string header = "3.0 " + Nickname + " " + destination + '\n';

            byte[] datagram = new byte[header.Length + data.Length];

            Encoding.ASCII.GetBytes(header, 0, header.Length, datagram, 0);

            Array.Copy(data, 0, datagram, header.Length, data.Length);

            return await udpClient.SendAsync(datagram, datagram.Length);
        }

        public async Task<ReceivedDatagram> Receive()
        {
            Reply reply = await samClient.ReceiveReply();

            if ((!rawDatagrams && reply.Description != "DATAGRAM RECEIVED") ||
                (rawDatagrams && reply.Description != "RAW RECEIVED"))
                throw new Exception("Invalid SAM reply.");

            string destination = null;

            if (!rawDatagrams)
                destination = reply.Properties["DESTINATION"];

            int size = Convert.ToInt32(reply.Properties["SIZE"]);

            if (size < 0 || size > 32768)
                throw new Exception("Invalid SAM reply.");

            byte[] data = new byte[size];

            await samClient.TcpClient.GetStream().ReadAsync(data, 0, data.Length);

            return new ReceivedDatagram(destination, data);
        }

        public bool Alive
        {
            get
            {
                return samClient.TcpClient.Connected;
            }
        }
    }

    public class ReceivedDatagram
    {
        public string Destination { get; private set; }
        public byte[] Data { get; private set; }

        internal ReceivedDatagram(string destination, byte[] data)
        {
            Destination = destination;
            Data = data;
        }
    }
}