﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.IO.IsolatedStorage;
using System.Linq;
using System.Xml;

namespace I2PTalk
{
    public enum AuthorizationPolicy
    {
        AcceptAll,
        Ask,
        BlockAll
    }

    public enum IncomingChatMessageAction
    {
        BlinkTrayIcon,
        ShowBalloon,
        ShowChatWindow
    }

    public class Settings
    {
        readonly string settingsFileName = "settings.xml";

        public Identity Identity { get; set; }

        public bool ConnectAtStartup { get; set; }
        public AuthorizationPolicy AuthorizationPolicy { get; set; }
        public TimeSpan StatusUpdateInterval { get; set; }
        public TimeSpan StatusUpdateTimeout { get; set; }

        public bool SaveWindowLocation { get; set; }
        public bool ShowTimeStamps { get; set; }
        public Point WindowLocation { get; set; }
        public IncomingChatMessageAction IncomingChatMessageAction { get; set; }

        public string SamHost { get; set; }
        public int SamPort { get; set; } 
        public Dictionary<string, string> SamOptions { get; set; }        

        public Settings()
        {
            ResetToDefault();
        }

        public bool Load()
        {
            return Load(settingsFileName);
        }

        public bool Load(string fileName)
        {
            try
            {
                //using (IsolatedStorageFile isFile = IsolatedStorageFile.GetUserStoreForApplication())
                using (IsolatedStorageFileStream isFileStream = new IsolatedStorageFileStream(settingsFileName, FileMode.Open, FileAccess.Read))
                using (XmlReader xmlReader = XmlReader.Create(isFileStream))
                {
                    if (xmlReader.MoveToContent() != XmlNodeType.Element || xmlReader.Name != "I2PTalkSettings" || xmlReader.GetAttribute("Version") != "1.0")
                        return false;

                    xmlReader.Read();

                    Identity = Identity.FromXml(xmlReader);

                    ConnectAtStartup = xmlReader.ReadElementContentAsBoolean("ConnectAtStartup", string.Empty);
                    string authorizationPolicy = xmlReader.ReadElementContentAsString("AuthorizationPolicy", string.Empty);
                    AuthorizationPolicy = (AuthorizationPolicy)Enum.Parse(typeof(AuthorizationPolicy), authorizationPolicy);
                    StatusUpdateInterval = TimeSpan.FromMilliseconds(xmlReader.ReadElementContentAsInt("StatusUpdateInterval", string.Empty));
                    StatusUpdateTimeout = TimeSpan.FromMilliseconds(xmlReader.ReadElementContentAsInt("StatusUpdateTimeout", string.Empty));

                    SaveWindowLocation = xmlReader.ReadElementContentAsBoolean("SaveWindowLocation", string.Empty);                    
                    string windowLocation = xmlReader.ReadElementContentAsString("WindowLocation", string.Empty);
                    string[] wl = windowLocation.Split(new char[] { ',' }, 2);
                    WindowLocation = new Point(Convert.ToInt32(wl[0]), Convert.ToInt32(wl[1]));
                    ShowTimeStamps = xmlReader.ReadElementContentAsBoolean("ShowTimeStamps", string.Empty);
                    string incomingChatMessageAction = xmlReader.ReadElementContentAsString("IncomingChatMessageAction", string.Empty);
                    IncomingChatMessageAction = (IncomingChatMessageAction)Enum.Parse(typeof(IncomingChatMessageAction), incomingChatMessageAction);
                    
                    SamHost = xmlReader.ReadElementContentAsString("SamHost", string.Empty);
                    SamPort = xmlReader.ReadElementContentAsInt("SamPort", string.Empty);
                    SamOptions = GetSamOptionsFromString(xmlReader.ReadElementContentAsString("SamOptions", string.Empty));                    
                }

                return true;
            }
            catch
            {
                return false;
            }
        }

        private string GetSamOptionsAsString()
        {
            return string.Join(";", SamOptions.Select(option => option.Key + "=" + option.Value));
        }

        private Dictionary<string, string> GetSamOptionsFromString(string s)
        {
            Dictionary<string, string> options = new Dictionary<string, string>();

            foreach (string part in s.Split(';'))
            {
                if (part == string.Empty)
                    continue;

                string[] keyvalue = part.Split('=');

                if (keyvalue.Length != 2)
                    return new Dictionary<string,string>();

                options.Add(keyvalue[0], keyvalue[1]);
            }

            return options;
        }

        public bool Save()
        {
            return Save(settingsFileName);
        }

        public bool Save(string fileName)
        {
            try
            {
                //using (IsolatedStorageFile isFile = IsolatedStorageFile.GetMachineStoreForApplication())
                using (IsolatedStorageFileStream isFileStream = new IsolatedStorageFileStream(settingsFileName, FileMode.Create, FileAccess.Write))
                using (XmlWriter xmlWriter = XmlWriter.Create(isFileStream))
                {
                    xmlWriter.WriteStartDocument();
                    xmlWriter.WriteStartElement("I2PTalkSettings");
                    xmlWriter.WriteAttributeString("Version", "1.0");

                    Identity.WriteToXml(xmlWriter);

                    xmlWriter.WriteElementBoolean("ConnectAtStartup", ConnectAtStartup);
                    xmlWriter.WriteElementString("AuthorizationPolicy", Enum.GetName(typeof(AuthorizationPolicy), AuthorizationPolicy));
                    xmlWriter.WriteElementInt("StatusUpdateInterval", (int)StatusUpdateInterval.TotalMilliseconds);
                    xmlWriter.WriteElementInt("StatusUpdateTimeout", (int)StatusUpdateTimeout.TotalMilliseconds);

                    xmlWriter.WriteElementBoolean("SaveWindowLocation", SaveWindowLocation);
                    xmlWriter.WriteElementString("WindowLocation", string.Format("{0},{1}", WindowLocation.X, WindowLocation.Y));
                    xmlWriter.WriteElementBoolean("ShowTimeStamps", ShowTimeStamps);
                    xmlWriter.WriteElementString("IncomingChatMessageAction", Enum.GetName(typeof(IncomingChatMessageAction), IncomingChatMessageAction));

                    xmlWriter.WriteElementString("SamHost", SamHost);
                    xmlWriter.WriteElementInt("SamPort", SamPort);
                    xmlWriter.WriteElementString("SamOptions", GetSamOptionsAsString());

                    xmlWriter.WriteEndElement();
                    xmlWriter.WriteEndDocument();
                }

                return true;
            }
            catch
            {
                return false;
            }
        }

        public void ResetToDefault()
        {
            Identity = null;

            ConnectAtStartup = false;
            AuthorizationPolicy = AuthorizationPolicy.Ask;
            StatusUpdateInterval = TimeSpan.FromSeconds(20);
            StatusUpdateTimeout = TimeSpan.FromSeconds(75);

            SaveWindowLocation = true;
            WindowLocation = Point.Empty;
            ShowTimeStamps = true;
            IncomingChatMessageAction = IncomingChatMessageAction.ShowChatWindow;

            SamHost = "localhost";
            SamPort = 7656;
            SamOptions = new Dictionary<string, string>();
            SamOptions.Add("inbound.length", "3");
            SamOptions.Add("outbound.length", "3");
            SamOptions.Add("inbound.lengthVariance", "0");
            SamOptions.Add("outbound.lengthVariance", "0");
            SamOptions.Add("inbound.quantity", "2");
            SamOptions.Add("outbound.quantity", "2");
            SamOptions.Add("inbound.backupQuantity", "0");
            SamOptions.Add("outbound.backupQuantity", "0");
        }
    }
}
