﻿using SamLib;
using System.IO;
using System.Text;
using System.Threading.Tasks;

namespace I2PTalk
{
    public enum MessageType
    {
        StatusUpdate,
        ChatMessage,
    }

    public interface IMessage
    {
        void Write(BinaryWriter binaryWriter);
    }

    public class StatusUpdateMessage : IMessage
    {
        public string ContactName { get; private set; }
        public ContactStatus ContactStatus { get; private set; }
        public string Profile { get; private set; }
        public string AuthorizationMessage { get; private set; }

        public StatusUpdateMessage(string contactName, ContactStatus contactStatus, string profile, string authorizationMessage)
        {
            ContactName = contactName;
            ContactStatus = contactStatus;
            Profile = profile;
            AuthorizationMessage = authorizationMessage;
        }

        public static StatusUpdateMessage Read(BinaryReader binaryReader)
        {
            try
            {
                string contactName = binaryReader.ReadString();
                ContactStatus contactStatus = (ContactStatus)binaryReader.ReadByte();
                string profile = binaryReader.ReadString();
                string authorizationMessage = binaryReader.ReadString();

                if (authorizationMessage == string.Empty)
                    authorizationMessage = null;

                if (binaryReader.BaseStream.Position < binaryReader.BaseStream.Length)
                    return null;

                return new StatusUpdateMessage(contactName, contactStatus, profile, authorizationMessage);
            }
            catch
            {
                return null;
            }
        }

        public void Write(BinaryWriter binaryWriter)
        {
            binaryWriter.Write((byte)MessageType.StatusUpdate);
            binaryWriter.Write(ContactName);
            binaryWriter.Write((byte)ContactStatus);
            binaryWriter.Write(Profile);
            binaryWriter.Write(AuthorizationMessage ?? string.Empty);
        }
    }

    public class ChatMessage : IMessage
    {
        public string Message { get; private set; }

        public ChatMessage(string message)
        {
            Message = message;
        }

        public static ChatMessage Read(BinaryReader binaryReader)
        {
            try
            {
                string message = binaryReader.ReadString();

                if (binaryReader.BaseStream.Position < binaryReader.BaseStream.Length)
                    return null;

                return new ChatMessage(message);
            }
            catch
            {
                return null;
            }
        }

        public void Write(BinaryWriter binaryWriter)
        {
            binaryWriter.Write((byte)MessageType.ChatMessage);
            binaryWriter.Write(Message);
        }
    }   

    public class Protocol
    {
        DatagramSession datagramSession;

        public static readonly byte Version = 1;

        public Protocol(DatagramSession datagramSession)
        {
            this.datagramSession = datagramSession;
        }

        private async Task SendMessage(Contact contact, IMessage message)
        {
            MemoryStream memoryStream = new MemoryStream();

            BinaryWriter binaryWriter = new BinaryWriter(memoryStream, Encoding.Unicode);

            binaryWriter.Write(Version);

            message.Write(binaryWriter);

            byte[] buffer = new byte[memoryStream.Position];

            memoryStream.Seek(0, SeekOrigin.Begin);
            memoryStream.Read(buffer, 0, buffer.Length);

            binaryWriter.Close();

            await datagramSession.Send(contact.Destination, buffer);
        }

        private async Task SendMessage(Contact contact, MessageType messageType)
        {
            byte[] buffer = new byte[] { Version, (byte)messageType };

            await datagramSession.Send(contact.Destination, buffer);
        }

        public async Task SendStatusUpdateMessage(Contact contact, string contactName, ContactStatus contactStatus, string profile, string authorizationMessage)
        {
            await SendMessage(contact, new StatusUpdateMessage(contactName, contactStatus, profile, authorizationMessage));
        }

        public async Task SendChatMessage(Contact contact, string message)
        {
            await SendMessage(contact, new ChatMessage(message));
        }
    }
}
