﻿using I2PTalk.Properties;
using System;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace I2PTalk
{
    public partial class MainForm : Form
    {
        Core core;

        public static readonly Icon[] NotifyIcons = new Icon[] {
            Resources.notify_status_offline,
            Resources.notify_status_online,
            Resources.notify_status_away,
            Resources.notify_status_busy,
            Resources.notify_status_offline };

        public MainForm()
        {
            InitializeComponent();

            core = new Core(this);

            if (core.Settings.SaveWindowLocation)
                Location = core.Settings.WindowLocation;
            else
                StartPosition = FormStartPosition.CenterScreen;

            if (core.Settings.ConnectAtStartup)
                toolStripButtonConnect_Click(this, new EventArgs());

            Update();
        }

        //private delegate ChatForm CreateNewChatFormDelegate(Contact contact);

        //public ChatForm CreateNewChatForm(Contact contact)
        //{
        //    if (InvokeRequired)
        //        return (ChatForm)Invoke(new CreateNewChatFormDelegate(CreateNewChatForm), contact);

        //    return new ChatForm(core, contact);
        //}

        private delegate void UpdateDelegate();

        public new void Update()
        {
            if (InvokeRequired)
            {
                Invoke(new UpdateDelegate(Update));
                return;
            }

            bool unreadMessagesAvailable = false;

            for (int i = 0; i < core.Identity.Contacts.Count; i++)
            {
                Contact contact = core.Identity.Contacts[i];
                ListViewItem listViewItem = listViewContacts.Items.OfType<ListViewItem>().FirstOrDefault(lvi => lvi.Tag == contact);

                bool hasUnreadMessages = core.ChatForms.ContainsKey(contact) && core.ChatForms[contact].HasUnreadMessages;

                StringBuilder sbToolTipText = new StringBuilder();
                sbToolTipText.Append(contact.DisplayName);
                if (contact.Name != contact.DisplayName)
                    sbToolTipText.AppendFormat(" ({0})", contact.Name);
                sbToolTipText.AppendLine();
                sbToolTipText.AppendLine();
                if (!contact.PendingAuthorizationRequest)
                    sbToolTipText.AppendFormat("Status: {0}", Enum.GetName(typeof(ContactStatus), contact.Status));
                else
                    sbToolTipText.Append("Status: Waiting for authorization");
                if (hasUnreadMessages)
                {
                    sbToolTipText.AppendLine();
                    sbToolTipText.AppendLine("Unread messages available");
                }

                string toolTipText = sbToolTipText.ToString();

                if (listViewItem == null)
                {
                    listViewItem = new ListViewItem();
                    listViewItem.Text = contact.DisplayName;
                    if (hasUnreadMessages)
                    {
                        listViewItem.ImageIndex = 4;
                        unreadMessagesAvailable = true;
                    }
                    else
                        if (contact.PendingAuthorizationRequest)
                            listViewItem.ImageIndex = 5;
                        else
                            listViewItem.ImageIndex = (int)contact.Status;
                    listViewItem.ToolTipText = toolTipText;
                    listViewItem.Tag = contact;
                    listViewContacts.Items.Add(listViewItem);
                }
                else
                {
                    if (listViewItem.Text != contact.DisplayName)
                        listViewItem.Text = contact.DisplayName;
                    if (hasUnreadMessages)
                    {
                        if (listViewItem.ImageIndex != 4)
                            listViewItem.ImageIndex = 4;
                        unreadMessagesAvailable = true;
                    }
                    else
                        if (contact.PendingAuthorizationRequest)
                        {
                            if (listViewItem.ImageIndex != 5)
                                listViewItem.ImageIndex = 5;
                        }
                        else
                            if (listViewItem.ImageIndex != (int)contact.Status)
                                listViewItem.ImageIndex = (int)contact.Status;
                    if (listViewItem.ToolTipText != toolTipText)
                        listViewItem.ToolTipText = toolTipText;
                }   
            }

            if (core.Identity.AuthorizationRequests.Count == 0)
            {
                if (panelAuthorizationRequest.Visible)
                    panelAuthorizationRequest.Visible = false;
            }
            else
            {
                labelAuthorizationRequest.Text = string.Format("{0} pending authorization request{1}...", core.Identity.AuthorizationRequests.Count, core.Identity.AuthorizationRequests.Count == 1 ? string.Empty : "s");
                panelAuthorizationRequest.Visible = true;
            }

            for (int i = 0; i < listViewContacts.Items.Count; i++)
                if (core.Identity.Contacts.IndexOf((Contact)listViewContacts.Items[i].Tag) == -1)
                    listViewContacts.Items.RemoveAt(i);

            if (toolStripDropDownButtonStatus.Tag == null || (ContactStatus)toolStripDropDownButtonStatus.Tag != core.OwnStatus)
            {
                switch (core.OwnStatus)
                {
                    case ContactStatus.Online:
                        toolStripDropDownButtonStatus.Image = Resources.status_online;
                        toolStripDropDownButtonStatus.Text = "Online";
                        break;
                    case ContactStatus.Away:
                        toolStripDropDownButtonStatus.Image = Resources.status_away;
                        toolStripDropDownButtonStatus.Text = "Away";
                        break;
                    case ContactStatus.Busy:
                        toolStripDropDownButtonStatus.Image = Resources.status_busy;
                        toolStripDropDownButtonStatus.Text = "Busy";
                        break;
                }
                onlineToolStripMenuItem1.Checked = core.OwnStatus == ContactStatus.Online;
                awayToolStripMenuItem1.Checked = core.OwnStatus == ContactStatus.Away;
                busyToolStripMenuItem1.Checked = core.OwnStatus == ContactStatus.Busy;
                invisibleToolStripMenuItem1.Checked = core.OwnStatus == ContactStatus.Invisible;               
            }

            toolStripDropDownButtonStatus.Tag = core.OwnStatus;

            if (unreadMessagesAvailable)
            {
                notifyIcon.Text = "I2PTalk - unread messages available";
                notifyIcon.Icon = Resources.notify_incoming_chat_message;
            }
            else
            {
                notifyIcon.Text = "I2PTalk";
                if (core.SamStatus == SamStatus.Connected)
                    notifyIcon.Icon = NotifyIcons[(int)core.OwnStatus];
                else
                    notifyIcon.Icon = Resources.notify_status_offline;
            }

            switch (core.SamStatus)
            {
                case SamStatus.Connected:
                    toolStripButtonConnect.Enabled = false;
                    toolStripButtonDisconnect.Enabled = true;
                    toolStripDropDownButtonStatus.Enabled = true;
                    toolStripStatusLabelStatus.Text = string.Format("Connected as \"{0}\".", core.Identity.Name);
                    break;
                case SamStatus.Connecting:
                    toolStripButtonConnect.Enabled = false;
                    toolStripButtonDisconnect.Enabled = true;
                    toolStripDropDownButtonStatus.Enabled = false;
                    toolStripStatusLabelStatus.Text = "Connecting...";
                    break;
                case SamStatus.Disconnected:
                    toolStripButtonConnect.Enabled = true;
                    toolStripButtonDisconnect.Enabled = false;
                    toolStripDropDownButtonStatus.Enabled = false;
                    toolStripStatusLabelStatus.Text = "Disconnected.";
                    break;
            }
        }

        private async void onlineToolStripMenuItem_Click(object sender, EventArgs e)
        {
            await core.ChangeOwnStatus(ContactStatus.Online);
        }

        private async void awayToolStripMenuItem_Click(object sender, EventArgs e)
        {
            await core.ChangeOwnStatus(ContactStatus.Away);
        }

        private async void busyToolStripMenuItem_Click(object sender, EventArgs e)
        {
            await core.ChangeOwnStatus(ContactStatus.Busy);
        }

        private void invisibleToolStripMenuItem_Click(object sender, EventArgs e)
        {
            MessageBox.Show("Not implemented yet. ;)");
        }

        private async void toolStripButtonConnect_Click(object sender, EventArgs e)
        {
            await core.SamConnect();
        }

        private async void toolStripButtonDisconnect_Click(object sender, EventArgs e)
        {
            await core.SamDisconnect();
        }

        private void copyMyAddressToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (!string.IsNullOrEmpty(core.Identity.Destination))
            {
                Clipboard.SetText(core.Identity.GetI2PTalkAddress());

                MessageBox.Show("Your I2PTalk address has been copied to your clipboard.", "Copy own address", MessageBoxButtons.OK, MessageBoxIcon.Information);
            }
            else
                MessageBox.Show("Please connect a first time to have an address generated for you.", "Copy own address", MessageBoxButtons.OK, MessageBoxIcon.Information);
        }

        private void contextMenuStripContacts_Opening(object sender, CancelEventArgs e)
        {
            if (listViewContacts.SelectedItems.Count == 0)
            {
                chatToolStripMenuItem.Enabled = false;
                contactDetailsToolStripMenuItem.Enabled = false;
                renameToolStripMenuItem.Enabled = false;
                deleteAndBlockToolStripMenuItem.Enabled = false;
            }
            else
                if (core.SamStatus != SamStatus.Connected)
                {
                    chatToolStripMenuItem.Enabled = false;
                    contactDetailsToolStripMenuItem.Enabled = false;
                    renameToolStripMenuItem.Enabled = true;
                    deleteAndBlockToolStripMenuItem.Enabled = true;
                }
                else
                {
                    Contact contact = (Contact)listViewContacts.SelectedItems[0].Tag;

                    chatToolStripMenuItem.Enabled = contact.Status != ContactStatus.Offline;
                    contactDetailsToolStripMenuItem.Enabled = contact.Status != ContactStatus.Offline;
                    renameToolStripMenuItem.Enabled = true;
                    deleteAndBlockToolStripMenuItem.Enabled = true;
                }
        }

        private void toolStripButtonAddContact_Click(object sender, EventArgs e)
        {
            core.AddContact();
        }

        private void chatToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Contact contact = (Contact)listViewContacts.SelectedItems[0].Tag;

            ChatForm chatForm;

            if (core.ChatForms.ContainsKey(contact))
                chatForm = core.ChatForms[contact];
            else
            {
                chatForm = new ChatForm(core, contact);
                core.ChatForms.Add(contact, chatForm);
            }

            chatForm.Show();
            if (chatForm.WindowState == FormWindowState.Minimized)
                chatForm.WindowState = FormWindowState.Normal;
            chatForm.Activate();

            if (chatForm.HasUnreadMessages)
            {
                chatForm.HasUnreadMessages = false;
                Update();
            }
        }

        private void contactDetailsToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Contact contact = (Contact)listViewContacts.SelectedItems[0].Tag;

            //await core.ShowContactDetails(contact);
        }

        private void renameToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Contact contact = (Contact)listViewContacts.SelectedItems[0].Tag;

            core.RenameContact(contact);
        }

        private void deleteAndBlockToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Contact contact = (Contact)listViewContacts.SelectedItems[0].Tag;

            core.DeleteAndBlockContact(contact);
        }

        private void addContactToolStripMenuItem_Click(object sender, EventArgs e)
        {
            toolStripButtonAddContact_Click(sender, e);
        }

        private async void identitiesToolStripMenuItem_Click(object sender, EventArgs e)
        {
            MyIdentityForm myIdentityForm = new MyIdentityForm(core.Identity);

            if (myIdentityForm.ShowDialog() != DialogResult.OK)
                return;

            if (myIdentityForm.IdentityChanged)
            {
                await core.ChangeIdentity(myIdentityForm.Identity);                
            }
        }

        private async void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (e.CloseReason == CloseReason.UserClosing)
                if (MessageBox.Show("Are you sure you want to close I2PTalk?", "Close I2PTalk?", MessageBoxButtons.OKCancel, MessageBoxIcon.Information) != DialogResult.OK)
                {
                    e.Cancel = true;
                    return;
                }

            core.Settings.WindowLocation = Location;

            await core.Shutdown();
        }

        private void contextMenuStripNotifyIcon_Opening(object sender, CancelEventArgs e)
        {
            openToolStripMenuItem.Enabled = WindowState == FormWindowState.Minimized;
            statusToolStripMenuItem.Enabled = toolStripDropDownButtonStatus.Enabled;
            connectToolStripMenuItem.Enabled = toolStripButtonConnect.Enabled;
            disconnectToolStripMenuItem.Enabled = toolStripButtonDisconnect.Enabled;
        }

        private void openToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Show();
            WindowState = FormWindowState.Normal;            
        }

        private void quitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void MainForm_Resize(object sender, EventArgs e)
        {
            if (WindowState == FormWindowState.Minimized)
                Hide();
        }

        private void toolStripButtonSettings_Click(object sender, EventArgs e)
        {
            SettingsForm settingsForm = new SettingsForm(core);

            settingsForm.ShowDialog();
        }

        private void notifyIcon_DoubleClick(object sender, EventArgs e)
        {
            if (WindowState == FormWindowState.Minimized)
            {
                bool openMainForm = true;

                foreach (ChatForm chatForm in core.ChatForms.Values)
                    if (chatForm.HasUnreadMessages)
                    {
                        chatForm.Show();
                        chatForm.HasUnreadMessages = false;
                        openMainForm = false;
                    }

                if (openMainForm)
                    openToolStripMenuItem_Click(this, new EventArgs());
                else
                    Update();
            }
            else
                WindowState = FormWindowState.Minimized;
        }

        private void listViewContacts_DoubleClick(object sender, EventArgs e)
        {
            if (listViewContacts.SelectedItems.Count != 1)
                return;

            Contact contact = (Contact)listViewContacts.SelectedItems[0].Tag;

            ChatForm chatForm;

            if (core.ChatForms.ContainsKey(contact))
                chatForm = core.ChatForms[contact];
            else
            {
                chatForm = new ChatForm(core, contact);
                core.ChatForms.Add(contact, chatForm);
            }

            chatForm.Show();
            if (chatForm.WindowState == FormWindowState.Minimized)
                chatForm.WindowState = FormWindowState.Normal;
            chatForm.Activate();

            if (chatForm.HasUnreadMessages)
            {
                chatForm.HasUnreadMessages = false;
                Update();
            }
        }

        private void labelAuthorizationRequest_Click(object sender, EventArgs e)
        {
            AuthorizationRequestsForm authorizationRequestsForm = new AuthorizationRequestsForm(core.Identity);

            if (authorizationRequestsForm.ShowDialog() != DialogResult.OK)
                return;

            Update();
        }

        private void toolStripButtonInfo_Click(object sender, EventArgs e)
        {
            MessageBox.Show("I2PTalk 1.0 alpha" + Environment.NewLine +
                Environment.NewLine +
                "Uses parts of the Silk icon set (http://www.famfamfam.com/lab/icons/silk/)");
        }
    }
}