﻿using SamLib;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace I2PTalk
{
    public class Core
    {
        public Settings Settings { get; private set; }
        
        public SamStatus SamStatus; // do we need this?

        public Identity Identity { get; private set; }

        public ContactStatus OwnStatus { get; private set; }

        public Protocol Protocol { get; private set; }
  
        public Dictionary<Contact, ChatForm> ChatForms { get; private set; }

        MainForm mainForm;

        DatagramSession datagramSession;

        public Core(MainForm mainForm)
        {
            this.mainForm = mainForm;

            Settings = new Settings();

            if (Settings.Load())
                Identity = Settings.Identity;
            else
            {
                Settings.ResetToDefault();

                NewIdentityForm newIdentityForm = new NewIdentityForm(true);

                if (newIdentityForm.ShowDialog() != DialogResult.OK)
                {
                    Environment.Exit(0);
                    return;
                }

                Identity = newIdentityForm.Identity;
            }

            SamStatus = SamStatus.Disconnected;

            OwnStatus = ContactStatus.Online;

            ChatForms = new Dictionary<Contact, ChatForm>();
        }

        public async Task SamConnect()
        {
            SamStatus = SamStatus.Connecting;

            mainForm.Update();

            datagramSession = new DatagramSession("I2PTalk" + new Random().Next(10000), false, Identity.Destination, Settings.SamOptions);

            Protocol = new Protocol(datagramSession);

            try
            {
                await datagramSession.Open(Settings.SamHost, Settings.SamPort, Settings.SamPort - 1); // should have SamUdpPort setting
            }
            catch (SamException e)
            {
                datagramSession.Close();
                datagramSession = null;
                Protocol = null;

                SamStatus = SamStatus.Disconnected;

                mainForm.Update();

                MessageBox.Show(string.Format("I2PTalk was able to connect to the SAM bridge of your router, however it could not open a session (error code: {0}, message: {1}).", e.SamError, e.SamMessage), "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
               
                return;
            }
            catch (SocketException e)
            {
                datagramSession.Close();
                datagramSession = null;
                Protocol = null;

                SamStatus = SamStatus.Disconnected;

                mainForm.Update();

                MessageBox.Show(string.Format("I2PTalk was unable to connect to the SAM bridge of your router (error code: {0}).\r\n" +
                                              "Make sure SAM is started (go to http://localhost:7657/configclients) and that your router settings in I2PTalk are correct.", e.Message), "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);

                return;
            }

            SamStatus = SamStatus.Connected;

            Identity.Destination = datagramSession.Destination;

            Task receiveWorker = ReceiveWorker();
            Task contactUpdateWorker = ContactUpdateWorker();

            mainForm.Update();
        }

        public async Task SamDisconnect()
        {
            if (SamStatus != SamStatus.Connected)
                return;

            foreach (Contact contact in Identity.Contacts)
                if (contact.Status != ContactStatus.Offline)
                {
                    await Protocol.SendStatusUpdateMessage(contact, Identity.Name, ContactStatus.Offline, Identity.Profile, null);
                    contact.Status = ContactStatus.Offline;
                }

            foreach (ChatForm chatForm in ChatForms.Values)
                chatForm.Close();

            ChatForms.Clear();

            for (int i = 0; i < Identity.Contacts.Count; i++)
                Identity.Contacts[i].Status = ContactStatus.Offline;

            datagramSession.Close();

            SamStatus = SamStatus.Disconnected;

            mainForm.Update();
        }

        public async Task ChangeOwnStatus(ContactStatus status)
        {
            OwnStatus = status;

            foreach (Contact contact in Identity.Contacts)
                if (contact.Status != ContactStatus.Offline)
                    await Protocol.SendStatusUpdateMessage(contact, Identity.Name, OwnStatus, Identity.Profile, null);

            mainForm.notifyIcon.Icon = MainForm.NotifyIcons[(int)status];

            mainForm.Update();
        }

        public void AddContact()
        {
            AddContactForm addContactForm = new AddContactForm(this);

            if (addContactForm.ShowDialog() == DialogResult.OK)
            {
                Contact contact = Contact.CreateNew(addContactForm.ContactName, addContactForm.ContactDisplayName, 
                    addContactForm.ContactDestination, addContactForm.AuthorizationMessage, true);

                Identity.Contacts.Add(contact);
            }

            mainForm.Update();
        }

        public void RenameContact(Contact contact)
        {
            RenameContactForm renameContactForm = new RenameContactForm(this, contact.DisplayName);

            if (renameContactForm.ShowDialog() == DialogResult.OK)
            {
                contact.DisplayName = renameContactForm.ContactName;
                mainForm.Update();
            }
        }

        public void DeleteAndBlockContact(Contact contact)
        {
            if (MessageBox.Show(string.Format("Are you sure you want to delete \"{0}\" from your contact list? The contact will be added to the list of your blocked contacts.", contact.DisplayName), "Delete contact?", MessageBoxButtons.OKCancel, MessageBoxIcon.Information) == DialogResult.OK)
            {
                Identity.Contacts.Remove(contact);
                Identity.Blocklist.Add(new BlocklistEntry(contact.Name, contact.DisplayName, contact.Destination));

                if (ChatForms.ContainsKey(contact))
                {
                    ChatForms[contact].Close();
                    ChatForms.Remove(contact);
                }

                mainForm.Update();
            }
        }

        public async Task ChangeIdentity(Identity identity)
        {
            await SamDisconnect();

            Identity = identity;
            Settings.Identity = identity;

            mainForm.Update();

            MessageBox.Show(string.Format("Your new identity has been applied. You are now \"{0}\".", Identity.Name), "Identity applied", MessageBoxButtons.OK, MessageBoxIcon.Information);            
        }

        public async Task Shutdown()
        {
            await ChangeOwnStatus(ContactStatus.Offline);

            foreach (ChatForm chatForm in ChatForms.Values)
                chatForm.Close();

            ChatForms.Clear();

            if (datagramSession != null)
                datagramSession.Close();

            Settings.Identity = Identity;            
            Settings.Save();
        }

        private async Task ContactUpdateWorker()
        {
            while (true)
            {
                bool mainFormUpdate = false;

                DateTime now = DateTime.Now;

                var contactsTimedOut = Identity.Contacts.Where(c => c.Status != ContactStatus.Offline && now - c.LastHeard > Settings.StatusUpdateTimeout);

                foreach (Contact contact in contactsTimedOut)
                {
                    contact.Status = ContactStatus.Offline;
                    mainFormUpdate = true;
                }

                if (mainFormUpdate)
                    mainForm.Update();

                var contactsUpdateNeeded = Identity.Contacts.Where(c => c.LastStatusUpdateSent == null || now - c.LastStatusUpdateSent > Settings.StatusUpdateInterval);

                Contact firstContact = contactsUpdateNeeded.FirstOrDefault();

                if (firstContact != null)
                {
                    firstContact.LastStatusUpdateSent = now;
                    await Protocol.SendStatusUpdateMessage(firstContact, Identity.Name, OwnStatus, Identity.Profile, firstContact.AuthorizationMessage);
                }

                await Task.Delay(TimeSpan.FromSeconds(4));
            }
        }

        private async Task ReceiveWorker()
        {
            while (true)
            {
                ReceivedDatagram datagram;

                try
                {
                    datagram = await datagramSession.Receive();
                }
                catch (ObjectDisposedException)
                {
                    break;
                }

                if (datagram.Destination == null)
                {
                    Debug.Print("Anonymous datagram received.");
                    continue;
                }

                Debug.Print("Datagram received. Length: {0}", datagram.Data.Length);

                if (Identity.Blocklist.Any(be => be.ContactDestination == datagram.Destination))
                {
                    Debug.Print("Datagram from blocked destination received. Length: {0}", datagram.Data.Length);
                    return;
                }

                Contact contact = Identity.Contacts.FirstOrDefault(c => c.Destination == datagram.Destination);

                if (contact == null)
                {
                    contact = new Contact();
                    contact.Destination = datagram.Destination;
                }

                contact.LastHeard = DateTime.Now;

                ProcessReceivedData(contact, datagram.Data);
            }
        }

        private void ProcessReceivedData(Contact contact, byte[] data)
        {
            MemoryStream dataStream = new MemoryStream(data, false);

            BinaryReader binaryReader = new BinaryReader(dataStream, Encoding.Unicode);

            if (dataStream.Length == 0)
            {
                Debug.Print("Received empty packet. Ignoring.");
                return;
            }

            if (binaryReader.ReadByte() != Protocol.Version)
            {
                Debug.Print("Received message with incompatible protocol version. Ignoring.");
                return;
            }

            bool mainFormUpdate = false;

            MessageType messageType = (MessageType)binaryReader.ReadByte();

            switch (messageType)
            {
                case MessageType.StatusUpdate:
                    Debug.Print("Status update received.");

                    StatusUpdateMessage statusUpdateMessage = StatusUpdateMessage.Read(binaryReader);

                    if (statusUpdateMessage == null)
                        goto dataCorrupted;

                    if (contact.Name == null)
                    {
                        if (Identity.AuthorizationRequests.Any(ar => ar.ContactDestination == contact.Destination)) // we should store and update profile in authreq here
                            break;

                        switch (Settings.AuthorizationPolicy)
                        {
                            case AuthorizationPolicy.AcceptAll:
                                contact.LastStatusUpdateSent = DateTime.Now;
                                contact.Name = statusUpdateMessage.ContactName;
                                contact.DisplayName = statusUpdateMessage.ContactName;
                                contact.Status = statusUpdateMessage.ContactStatus;
                                Identity.Contacts.Add(contact);
                                mainFormUpdate = true;
                                break;
                            case AuthorizationPolicy.Ask:
                                Identity.AuthorizationRequests.Add(new AuthorizationRequest(statusUpdateMessage.ContactName, contact.Destination, statusUpdateMessage.AuthorizationMessage)); // here profile too
                                mainFormUpdate = true;
                                break;
                            case AuthorizationPolicy.BlockAll:
                                break;
                        }
                    }
                    else
                    {
                        if (contact.Status != statusUpdateMessage.ContactStatus)
                            mainFormUpdate = true;

                        contact.Status = statusUpdateMessage.ContactStatus;

                        if (contact.PendingAuthorizationRequest)
                        {
                            contact.PendingAuthorizationRequest = false;
                            contact.AuthorizationMessage = null;
                            mainFormUpdate = true;
                        }

                        
                    }

                    if (mainFormUpdate)
                        mainForm.Update();
                    break;
                case MessageType.ChatMessage:
                    Debug.Print("Chat message received.");
                    ChatMessage chatMessage = ChatMessage.Read(binaryReader);

                    if (chatMessage == null)
                        goto dataCorrupted;

                    ChatForm chatForm = null;

                    if (ChatForms.ContainsKey(contact))
                        chatForm = ChatForms[contact];

                    if (chatForm == null || !chatForm.Visible)
                    {
                        if (chatForm == null)
                        {
                            chatForm = //mainForm.CreateNewChatForm(connection.Contact);
                                       new ChatForm(this, contact);
                            ChatForms.Add(contact, chatForm);
                        }

                        chatForm.AddReceivedMessage(chatMessage.Message);

                        switch (Settings.IncomingChatMessageAction)
                        {
                            case IncomingChatMessageAction.BlinkTrayIcon:
                                chatForm.HasUnreadMessages = true;
                                mainForm.Update();
                                break;
                            case IncomingChatMessageAction.ShowBalloon:
                                mainForm.notifyIcon.ShowBalloonTip(60000, contact.DisplayName, chatMessage.Message, ToolTipIcon.Info);
                                break;
                            case IncomingChatMessageAction.ShowChatWindow:
                                chatForm.InvokeShow();
                                break;
                        }
                    }
                    else
                    {
                        chatForm.AddReceivedMessage(chatMessage.Message);

                        if (chatForm.WindowState == FormWindowState.Minimized)
                            chatForm.WindowState = FormWindowState.Normal; // replace with window flash?
                    }
                    break;
            }

            binaryReader.Close();

            return;

        dataCorrupted:
            Debug.Print("Received corrupted packet. Ignoring.");
        }
    }
}
