﻿using System;
using System.Collections.Generic;
using System.Xml;

namespace I2PTalk
{
    public enum ContactStatus
    {
        Offline,
        Online,
        Away,
        Busy,
        Invisible
    }

    public class Contact
    {
        public string Name { get; set; }
        public string DisplayName { get; set; }
        public string Destination { get; set; }
        public ContactStatus Status { get; set; }
        public DateTime LastHeard { get; set; }
        public DateTime? LastStatusUpdateSent { get; set; }
        public bool PendingAuthorizationRequest { get; set; }
        public string AuthorizationMessage { get; set; }

        public Contact()
        {
        }

        public static Contact CreateNew(string name, string displayName, string destination, string authorizationMessage, bool pendingAuthorizationRequest)
        {
            Contact contact = new Contact();

            contact.Name = name;
            contact.DisplayName = displayName;
            contact.Destination = destination;
            contact.Status = ContactStatus.Offline;
            contact.AuthorizationMessage = authorizationMessage;
            contact.PendingAuthorizationRequest = pendingAuthorizationRequest;

            return contact;
        }

        public string GetI2PTalkAddress()
        {
            return "i2ptalk:" + Name + ':' + Destination;
        }

        public static Contact FromXml(XmlReader xmlReader)
        {
            Contact contact = new Contact();

            xmlReader.ReadStartElement("Contact");
            contact.Name = xmlReader.ReadElementContentAsString("Name", string.Empty);
            contact.DisplayName = xmlReader.ReadElementContentAsString("DisplayName", string.Empty);
            contact.Destination = xmlReader.ReadElementContentAsString("Destination", string.Empty);
            contact.PendingAuthorizationRequest = xmlReader.ReadElementContentAsBoolean("PendingAuthorizationRequest", string.Empty);
            contact.AuthorizationMessage = xmlReader.ReadElementContentAsString("AuthorizationMessage", string.Empty);
            xmlReader.ReadEndElement();

            return contact;
        }

        public void WriteToXml(XmlWriter xmlWriter)
        {
            xmlWriter.WriteStartElement("Contact");
            xmlWriter.WriteElementString("Name", Name);
            xmlWriter.WriteElementString("DisplayName", DisplayName);
            xmlWriter.WriteElementString("Destination", Destination);
            xmlWriter.WriteElementBoolean("PendingAuthorizationRequest", PendingAuthorizationRequest);
            xmlWriter.WriteElementString("AuthorizationMessage", AuthorizationMessage);
            xmlWriter.WriteEndElement();
        }
    }

    public class BlocklistEntry
    {
        public string ContactName { get; private set; }
        public string ContactDisplayName { get; private set; }
        public string ContactDestination { get; private set; }

        public BlocklistEntry(string contactName, string contactDisplayName, string contactDestination)
        {
            ContactName = contactName;
            ContactDisplayName = contactDisplayName;
            ContactDestination = contactDestination;
        }

        public static BlocklistEntry FromXml(XmlReader xmlReader)
        {
            xmlReader.ReadStartElement("BlocklistEntry");
            string contactName = xmlReader.ReadElementContentAsString("ContactName", string.Empty);
            string contactDisplayName = xmlReader.ReadElementContentAsString("ContactDisplayName", string.Empty);
            string contactDestination = xmlReader.ReadElementContentAsString("ContactDestination", string.Empty);
            xmlReader.ReadEndElement();

            return new BlocklistEntry(contactName, contactDisplayName, contactDestination);
        }

        public void WriteToXml(XmlWriter xmlWriter)
        {
            xmlWriter.WriteStartElement("BlocklistEntry");
            xmlWriter.WriteElementString("ContactName", ContactName);
            xmlWriter.WriteElementString("ContactDisplayName", ContactDisplayName);
            xmlWriter.WriteElementString("ContactDestination", ContactDestination);           
            xmlWriter.WriteEndElement();
        }
    }

    public class AuthorizationRequest
    {
        public string ContactName { get; private set; }
        public string ContactDestination { get; private set; }
        public string Message { get; private set; }

        public AuthorizationRequest(string contactName, string contactDestination, string message)
        {
            ContactName = contactName;
            ContactDestination = contactDestination;
            Message = message;
        }

        public static AuthorizationRequest FromXml(XmlReader xmlReader)
        {
            xmlReader.ReadStartElement("AuthorizationRequest");
            string contactName = xmlReader.ReadElementContentAsString("ContactName", string.Empty);            
            string contactDestination = xmlReader.ReadElementContentAsString("ContactDestination", string.Empty);
            string message = xmlReader.ReadElementContentAsString("Message", string.Empty);
            xmlReader.ReadEndElement();

            return new AuthorizationRequest(contactName, contactDestination, message);
        }

        public void WriteToXml(XmlWriter xmlWriter)
        {
            xmlWriter.WriteStartElement("AuthorizationRequest");
            xmlWriter.WriteElementString("ContactName", ContactName);           
            xmlWriter.WriteElementString("ContactDestination", ContactDestination);
            xmlWriter.WriteElementString("Message", Message);
            xmlWriter.WriteEndElement();
        }
    }

    public class Identity
    {
        public string Name { get; private set; }
        public string Destination { get; set; }
        public string Profile { get; set; }
        public List<Contact> Contacts { get; private set; }
        public List<BlocklistEntry> Blocklist { get; private set; }
        public List<AuthorizationRequest> AuthorizationRequests { get; private set; }

        private Identity()
        {
        }

        public static Identity CreateNew(string name, string profile)
        {
            Identity identity = new Identity();

            identity.Name = name;
            identity.Profile = profile;
            identity.Contacts = new List<Contact>();
            identity.Blocklist = new List<BlocklistEntry>();
            identity.AuthorizationRequests = new List<AuthorizationRequest>();

            return identity;
        }

        public string GetI2PTalkAddress()
        {
            if (!string.IsNullOrEmpty(Destination))
                return "i2ptalk:" + Name + ':' + Destination.Substring(0, 516);
            else
                return string.Empty;
        }

        public static Identity FromXml(XmlReader xmlReader)
        {
            Identity identity = new Identity();

            xmlReader.ReadStartElement("Identity");
            identity.Name = xmlReader.ReadElementContentAsString("Name", string.Empty);
            identity.Destination = xmlReader.ReadElementContentAsString("Destination", string.Empty);
            identity.Profile = xmlReader.ReadElementContentAsString("Profile", string.Empty);

            //xmlReader.ReadStartElement("Contacts");
            //Contacts = new List<Contact>();
            //if (xmlReader.NodeType == XmlNodeType.Element && xmlReader.Name == "Contact")
            //{
            //    do
            //    {
            //        Contact contact = new Contact();
            //        contact.ReadFromXml(xmlReader);
            //        Contacts.Add(contact);
            //    } while (xmlReader.NodeType == XmlNodeType.Element && xmlReader.Name == "Contact");
            //    xmlReader.ReadEndElement();
            //}
            //xmlReader.ReadStartElement("Blocklist");
            //Blocklist = new Dictionary<string, string>();
            //if (xmlReader.NodeType == XmlNodeType.Element && xmlReader.Name == "BlocklistEntry")
            //{
            //    do
            //    {
            //        xmlReader.ReadStartElement("BlocklistEntry");
            //        string name = xmlReader.ReadElementContentAsString("Name", string.Empty);
            //        string destination = xmlReader.ReadElementContentAsString("Destination", string.Empty);
            //        xmlReader.ReadEndElement();
            //        Blocklist.Add(name, destination);
            //    } while (xmlReader.NodeType == XmlNodeType.Element && xmlReader.Name == "BlocklistEntry");
            //    xmlReader.ReadEndElement();
            //}
            //xmlReader.ReadStartElement("AuthorizationRequests");
            //AuthorizationRequests = new Dictionary<string, string>();      
            //if (xmlReader.NodeType == XmlNodeType.Element && xmlReader.Name == "AuthorizationRequest")
            //{
            //    do
            //    {
            //        xmlReader.ReadStartElement("AuthorizationRequest");
            //        string name = xmlReader.ReadElementContentAsString("Name", string.Empty);
            //        string message = xmlReader.ReadElementContentAsString("Message", string.Empty);
            //        xmlReader.ReadEndElement();
            //        Blocklist.Add(name, message);
            //    } while (xmlReader.NodeType == XmlNodeType.Element && xmlReader.Name == "AuthorizationRequest");
            //    xmlReader.ReadEndElement();
            //}
            //xmlReader.ReadEndElement();

            xmlReader.ReadStartElement("Contacts");
            identity.Contacts = new List<Contact>();
            while (xmlReader.NodeType == XmlNodeType.Element && xmlReader.Name == "Contact")
            {
                Contact contact = Contact.FromXml(xmlReader);
                identity.Contacts.Add(contact);
            }
            if (identity.Contacts.Count > 0)
                xmlReader.ReadEndElement();

            xmlReader.ReadStartElement("Blocklist");
            identity.Blocklist = new List<BlocklistEntry>();
            while (xmlReader.NodeType == XmlNodeType.Element && xmlReader.Name == "BlocklistEntry")
                identity.Blocklist.Add(BlocklistEntry.FromXml(xmlReader));
            if (identity.Blocklist.Count > 0)
                xmlReader.ReadEndElement();

            xmlReader.ReadStartElement("AuthorizationRequests");
            identity.AuthorizationRequests = new List<AuthorizationRequest>();
            while (xmlReader.NodeType == XmlNodeType.Element && xmlReader.Name == "AuthorizationRequest")
            {
                AuthorizationRequest authorizationRequest = AuthorizationRequest.FromXml(xmlReader);
                identity.AuthorizationRequests.Add(authorizationRequest);
            }
            if (identity.AuthorizationRequests.Count > 0)
                xmlReader.ReadEndElement();

            xmlReader.ReadEndElement();

            return identity;
        }

        public void WriteToXml(XmlWriter xmlWriter)
        {
            xmlWriter.WriteStartElement("Identity");
            xmlWriter.WriteElementString("Name", Name);
            xmlWriter.WriteElementString("Destination", Destination);
            xmlWriter.WriteElementString("Profile", Profile);

            xmlWriter.WriteStartElement("Contacts");
            foreach (Contact contact in Contacts)
                contact.WriteToXml(xmlWriter);
            xmlWriter.WriteEndElement();

            xmlWriter.WriteStartElement("Blocklist");
            foreach (BlocklistEntry blocklistEntry in Blocklist)
                blocklistEntry.WriteToXml(xmlWriter);
            xmlWriter.WriteEndElement();

            xmlWriter.WriteStartElement("AuthorizationRequests");
            foreach (AuthorizationRequest authorizationRequest in AuthorizationRequests)
                authorizationRequest.WriteToXml(xmlWriter);
            xmlWriter.WriteEndElement();

            xmlWriter.WriteEndElement();
        }
    }
}
