#! /usr/bin/env python
# -*- coding: utf-8 -*-
#-----------------------------------------------------------------------------
# Name:        graphpanel.py
# Purpose:     
#
# Author:      Jeremy Arendt
#
# Created:     2004/28/01
# RCS-ID:      $Id: graphpanel.py,v 1.2 2005/07/23 16:35:41 Inigo Exp $
# Copyright:   (c) 2002
# Licence:     See G3.LICENCE.TXT
#-----------------------------------------------------------------------------

import wx
from wx.lib import newevent
from random import randrange
from btconfig import BTConfig
from threading import Timer
from traceback import print_exc
from graph import Graph, GraphRecord
import time

TestUpdateEvent, EVT_TESTUPDATE = newevent.NewEvent()

class ColorLegendBox(wx.Panel):
    def __init__(self, parent, btconfig, name, colorfunc=None, size = (20,10)):
        wx.Panel.__init__(self, parent, -1, size=size, style = wx.SIMPLE_BORDER)
        self.color = btconfig.GetColor(name)
        self.name = name
        self.SetBackgroundColour(self.color)
        self.ColorFunc = colorfunc
        self.btconfig = btconfig
        wx.EVT_LEFT_DOWN(self, self.OnClick)
        wx.EVT_ERASE_BACKGROUND(self, self.Fill)
    
    def OnClick(self, event):
        color = self.GetColor(self.color)
        if color == None:
            return
        self.color = color
        self.SetBackgroundColour(self.color)
        self.btconfig.SetColor(self.name, self.color)
        self.btconfig.SaveOptions()
        if self.ColorFunc:
            self.ColorFunc(self.color, self.color)
        self.Refresh()
        
    def GetColor(self, color = wx.Color(0,0,0)):
        data = None
        c_dlg = wx.ColourDialog(self)
        c_dlg.GetColourData().SetChooseFull(True)
        c_dlg.GetColourData().SetColour(color)
        if c_dlg.ShowModal() == wx.ID_OK:
            data = c_dlg.GetColourData()
        c_dlg.Destroy()
        if data != None:
            return data.GetColour()
        else:
            return None

    def Fill(self, event):
        dc = event.GetDC()
        bgbrush = wx.Brush(self.color, wx.SOLID)
        dc.BeginDrawing()
        dc.SetBrush(bgbrush)
        dc.SetBackground(bgbrush)
        dc.Clear()
        dc.EndDrawing()

class GraphPanel(wx.Panel):
    def __init__(self, parent, btconfig, bmps):
        wx.Panel.__init__(self, parent, -1, style = wx.NO_FULL_REPAINT_ON_RESIZE, size=(600,600))
        self.count = 0
        self.btconfig = btconfig

        max_drate = (self.btconfig.Get('net_max_downrate') / 1024)
        max_urate = (self.btconfig.Get('net_max_uprate') / 1024)

        gp_total_drate1 = btconfig.GetColor('gp_total_drate1')
        gp_total_urate1 = btconfig.GetColor('gp_total_urate1')
        gp_sel_drate1 = btconfig.GetColor('gp_sel_drate1')
        gp_sel_urate1 = btconfig.GetColor('gp_sel_urate1')
        
        if max_drate == 0:
            max_drate = 200
        
        self.y_scale = max(max_drate, max_urate)
        self.x_scale = 600
        
        self.graph = Graph(self, btconfig, 15, _("time - %d minutes") % (10*max(0.25, self.btconfig.Get('gui_update_rate'))), 
                            _("scale - %d KBps") % (self.y_scale), size=(100,100))
        self.graph.Hide()
        
        self.total_urate = GraphRecord(gp_total_urate1, gp_total_urate1, self.x_scale, self.y_scale, 1)
        self.total_drate = GraphRecord(gp_total_drate1, gp_total_drate1, self.x_scale, self.y_scale, 1)
        self.sel_urate = GraphRecord(gp_sel_urate1, gp_sel_urate1, self.x_scale, self.y_scale, 1)
        self.sel_drate = GraphRecord(gp_sel_drate1, gp_sel_drate1, self.x_scale, self.y_scale, 1)
        
        self.attached = None
        
        self.graph.AddRecord(self.total_urate)
        self.graph.AddRecord(self.total_drate)
        
        self.SetBackgroundColour(btconfig.GetColor('gp_bgcolor'))
        
        label1 = wx.StaticText(self, -1, _("total down rate"))
        label2 = wx.StaticText(self, -1, _("total up rate"))
        label3 = wx.StaticText(self, -1, _("selected down rate"))
        label4 = wx.StaticText(self, -1, _("selected up rate"))
        
        clbox1 = ColorLegendBox(self, btconfig, 'gp_total_drate1', self.total_drate.ChangeColor)
        clbox2 = ColorLegendBox(self, btconfig, 'gp_total_urate1', self.total_urate.ChangeColor)
        clbox3 = ColorLegendBox(self, btconfig, 'gp_sel_drate1')
        clbox4 = ColorLegendBox(self, btconfig, 'gp_sel_urate1')

        colsizer = wx.FlexGridSizer(cols = 1)
        #colsizer = wx.BoxSizer(wx.VERTICAL)
        legsizer = wx.FlexGridSizer(cols = 8, hgap=2)
                
        legsizer.Add(clbox1, 0)
        legsizer.Add(label1, 0, wx.EXPAND)
        legsizer.Add(clbox2, 0)
        legsizer.Add(label2, 0, wx.EXPAND)
        legsizer.Add(clbox3, 0)
        legsizer.Add(label3, 0, wx.EXPAND)
        legsizer.Add(clbox4, 0)
        legsizer.Add(label4, 0, wx.EXPAND)
        legsizer.AddGrowableRow(0)
        legsizer.AddGrowableCol(1)
        legsizer.AddGrowableCol(3)
        legsizer.AddGrowableCol(5)
        legsizer.AddGrowableCol(7)
        
        colsizer.AddGrowableRow(0)
        colsizer.AddGrowableCol(0)

##        if wx.VERSION_STRING[:3] == "2.4":
##            ## wx 2.4
##            colsizer.Add(self.graph, 1, wx.EXPAND|wx.ADJUST_MINSIZE)
##            colsizer.Add(legsizer, 1, wx.EXPAND|wx.ADJUST_MINSIZE|wx.ALL, 4)       
##        else:
##            ## wx 2.5
        colsizer.Add(self.graph, 1, wx.EXPAND|wx.FIXED_MINSIZE)
        colsizer.Add(legsizer, 1, wx.EXPAND|wx.FIXED_MINSIZE|wx.ALL, 4)

        self.SetSizer(colsizer)
        self.Layout()

        EVT_TESTUPDATE(self, self.OnTestUpdate)
        ##EVT_ERASE_BACKGROUND(self, self.AntiFlicker)

    def AntiFlicker(self, event):
        pass
        
    def Show(self, value):
        self.graph.Show(value)
        
    def UpdateSize(self):
        self.graph.OnSize()
        self.Layout()
        
    def UpdateGraph(self):
        max_drate = (self.btconfig.Get('net_max_downrate') / 1024)
        max_urate = (self.btconfig.Get('net_max_uprate') / 1024)
        damax = max(max_drate, max_urate)
        
        if damax < 1:
            damax = 100
        
        if self.y_scale != damax:
            self.y_scale = damax
            self.graph.yaxis_text[0] = _("scale - %d KBps") % damax
            self.total_urate.SetYMax(damax, True)
            self.total_drate.SetYMax(damax, True)
            self.sel_urate.SetYMax(damax, True)
            self.sel_drate.SetYMax(damax, True)
        self.graph.UpdateGraph()
        
    def PushAsunder(self):
        if self.attached:
            self.graph.RemoveRecord(self.sel_drate)
            self.graph.RemoveRecord(self.sel_urate)
            self.attached = None
    
    def AttachSession(self, session):
        if session != None and session == self.attached:
            return
        elif session != None and session != self.attached:
            self.PushAsunder()
            
        self.attached = session
        drate_record, urate_record = session.GetRecords()
        
        self.sel_drate.cur_x = self.total_drate.cur_x
        self.sel_urate.cur_x = self.total_drate.cur_x
        x1 = self.total_drate.cur_x
        x2 = (self.x_scale - x1)        
        size = self.sel_drate.max_x
        
        self.sel_drate.line = [wx.Point(0,0)] * (size)
        self.sel_urate.line = [wx.Point(0,0)] * (size)
        self.sel_drate.line_data = [0] * (size)
        self.sel_urate.line_data = [0] * (size)
        
        self.sel_drate.line_data[0] = drate_record[0]
        self.sel_urate.line_data[0] = urate_record[0]
        
        self.sel_drate.line_data[1:x1+1] = drate_record[x2+1:]   #add current
        self.sel_urate.line_data[1:x1+1] = urate_record[x2+1:]   #add current
         
        self.sel_drate.line_data[x1+1:] = drate_record[1:x2+1]   #add old
        self.sel_urate.line_data[x1+1:] = urate_record[1:x2+1]   #add old

        self.sel_urate.SetYMax(self.y_scale)
        self.sel_drate.SetYMax(self.y_scale)
        
        self.graph.AddRecord(self.sel_drate)
        self.graph.AddRecord(self.sel_urate)
        
    def Append_Sel_DRate(self, value):
        if value == None:
            value = 0
        self.sel_drate.AddValue(value/1024)
    
    def Append_Sel_URate(self, value):
        if value == None:
            value = 0
        self.sel_urate.AddValue(value/1024)
        
    def Append_URate(self, value):
        if value == None:
            value = 0
        self.total_urate.AddValue(value/1024)
    
    def Append_DRate(self, value):
        if value == None:
            value = 0
        self.total_drate.AddValue(value/1024)

    def OnTestUpdate(self, event):
        if not hasattr(self, 'sess'):
            self.sess = DummySession()
            self.counter = 0
            self.AttachSession(self.sess)
        self.counter += 1
        
        if not self.counter % 10:
            self.AttachSession(self.sess)
        
        if not self.counter % 50:
            self.PushAsunder()
            
        r1 = randrange(0,20) * 1024
        r2 = randrange(0,20) * 1024
        r3 = (randrange(0,20)+20) * 1024
        r4 = (randrange(0,20)+20) * 1024
        
        self.Append_URate( r1 )
        self.Append_DRate( r2 )
        self.sess.AppendRateRecord(r1, r2)
        
        self.graph.UpdateGraph()
#---------------------------------------------------------------------------

class DummySession:
    def __init__(self):
        self.record_len = 601
        self.record_urate = [0] * self.record_len
        self.record_drate = [0] * self.record_len
        
    def GetId(self):
        return 100
    
    def GetRecords(self):
        return self.record_urate, self.record_drate
    
    def AppendRateRecord(self, drate, urate):
        if drate == None:
            drate = 0
        if urate == None:
            urate == 0
        self.record_drate.pop(0)
        self.record_urate.pop(0)
        self.record_drate.append(drate/1024)
        self.record_urate.append(drate/1024)
        
        
def RunTest(win):
    try:
        event = TestUpdateEvent()
        wx.PostEvent(win, event)
        timer = Timer(0.05, RunTest, [win])
        timer.start()
    except:
        print_exc()
        pass


def TestAttach():
    a = [0] * 20
    b = [0] * 10
    b += range(0,10)

    x1 = 10
    x2 = 20 - x1
        
    a[:x1] = b[x2:]   #add current
    a[x1:] = b[:x2]   #add old
    
    print a
    print b

if __name__ == "__main__":
    _ = lambda x: x # needed for gettext
    TestAttach()
    app = wx.PySimpleApp()
    frame = wx.Frame(None, -1, '', size=(600,400))
    btconfig = BTConfig()
    btconfig.LoadOptions()
    p = GraphPanel(frame, BTConfig(), None)

    p.Show(True)
    frame.Show(True)
    RunTest(p)

    app.MainLoop()



