#! /usr/bin/env python
# -*- coding: utf-8 -*-
#-----------------------------------------------------------------------------
# Name:        dlordermngr.py
# Purpose:     
#
# Author:      Jeremy Arendt
#
# Created:     2004/24/02
# RCS-ID:      $Id: dlordermngr.py,v 1.3 2005/09/12 22:36:18 Inigo Exp $
# Copyright:   (c) 2002
# Licence:     See G3.LICENCE
#-----------------------------------------------------------------------------
from wx.lib.mixins.listctrl import ColumnSorterMixin, ListCtrlAutoWidthMixin
from traceback import print_exc
from btconfig import BTConfig
from images import Images
from g3listctrl import *
from os.path import split
import wx

class DLOrderListCtrl(G3ListCtrl_wCheckBox, ListCtrlAutoWidthMixin, ColumnSorterMixin):
    def __init__(self, parent, btconfig, onrightclick, images):
        G3ListCtrl.__init__(self, parent, btconfig, "DLOrderMngr2")
        ListCtrlAutoWidthMixin.__init__(self)
        ColumnSorterMixin.__init__(self, 1)
        self.itemDataMap = {}
        self.OnRightClick = onrightclick
        self.images = images

        self.i_list = wx.ImageList(16, 16)
        self.SetImageList(self.i_list, wx.IMAGE_LIST_SMALL)
        
        wx.EVT_LIST_ITEM_RIGHT_CLICK(self, -1, self.OnRightClick)
        wx.EVT_LIST_COL_CLICK(self, -1, self.OnColClick)
        
    def OnColClick(self, event):
        pass    #this is not a user sorted list
        
    def GetListCtrl(self):
        return self
        
    def RePopulate(self):
        for i in range(0, len(self.itemData)):
            self.InsertRow(self.itemData[i][0], ["", "%d"%(self.itemData[i][0]), self.itemData[i][1], 
                                    self.itemData[i][2], self.itemData[i][3]])
            self.itemDataMap[self.itemData[i][0]] = [i+1]
        self.SortListItems(0, True)
    
    def Populate(self, file_ranges):
        self.itemData = []
        i = 0
        for file in file_ranges:
            name = split(file[0][0])[1]
            size = file[0][1]
            fraction = file[3] * 100
            self.InsertRow(i+1, ["", "%d"%(i+1), name, "%d KB"%(size/(1<<10)), "%0.1f"%fraction])
            self.Insert_Ebed_Ctrl(i+1, file[4])
            self.itemData.append([i+1, name, "%d KB"%(size/(1<<10)), "%0.1f"%fraction, file_ranges[i]])
            self.itemDataMap[i] = [i+1]
            i += 1
        
class DLOrderMngr(wx.Frame):
    """ Managment tool for multiple files in a torrent """
    def __init__(self, parent, btconfig, images, file_ranges, is_random, setRangesfunc):
        wx.Frame.__init__(self, parent, -1, _('MultiFile Torrent Manager'), size=(420,300) )
        
        if parent:
            pos = parent.GetPosition()
            self.SetPosition((pos[0]+100, pos[1]+50))
            
        self.is_random = is_random
        self.SetRanges = setRangesfunc
        self.file_ranges = file_ranges
        self.btconfig = btconfig
        
        panel = wx.Panel(self, -1)
        toolbar = wx.ToolBar(self, -1, size = (24,24), 
            style = wx.TB_FLAT)
        
        toolbar.SetToolBitmapSize((24,24))
        toolbar.AddTool(101, images.GetImage('upqueue24.png'), wx.NullBitmap, False, "", "Move up", "Move up")
        toolbar.AddTool(102, images.GetImage('downqueue24.png'), wx.NullBitmap, False, "", "Move down", "Move down")
        
        wx.EVT_TOOL(self, 101, self.OnMoveUp)
        wx.EVT_TOOL(self, 102, self.OnMoveDown)
        
        toolbar.Realize()
        self.toolbar = toolbar
        self.SetToolBar(toolbar)
        
        label1 = wx.StaticText(toolbar, -1, "This torrent contains multiple files.\nSelect the download order here, or leave it random")
        label1.SetPosition((100,2))
        
        radio_choices = [_("Random"), _("Custom Order")]
        
        rboxId = wx.NewId()
        self.radio1 = wx.RadioBox(panel, rboxId, _("Order Preference:"),
                        wx.DefaultPosition, wx.DefaultSize,
                        radio_choices, 1, wx.RA_SPECIFY_ROWS)
        if is_random:
            self.radio1.SetSelection(0)
            toolbar.EnableTool(101, False)
            toolbar.EnableTool(102, False)
        else:
            self.radio1.SetSelection(1)
            toolbar.EnableTool(101, True)
            toolbar.EnableTool(102, True)
        
        wx.EVT_RADIOBOX(panel, rboxId, self.OnRadio) 
        
        self.check1 = wx.CheckBox(panel, 301, _("Show this dialog any time a torrent contains multiple files"))
        self.check1.SetValue(btconfig['use_multimngr'])

        self.list = DLOrderListCtrl(panel, btconfig, self.OnRightClick, images)
        self.list.SetEbedCol(0)
        
        cols = [ [True, _("!"), wx.LIST_FORMAT_LEFT, 16],
                 [True, _("#"), wx.LIST_FORMAT_LEFT, 25],
                 [True, _("Filename"), wx.LIST_FORMAT_LEFT, 230],
                 [True, _("Size"), wx.LIST_FORMAT_RIGHT, 60],
                 [True, _("%"), wx.LIST_FORMAT_RIGHT, 50]
            ]

        self.list.InsertColumns(cols)

        button1 = wx.Button(panel, 201, _("OK"), size=(60,-1))
        button2 = wx.Button(panel, 202, _("Cancel"), size=(60,-1))
        
        wx.EVT_BUTTON(self, 201, self.OnButtonOk)
        wx.EVT_BUTTON(self, 202, self.OnButtonCancel)
        
        buttongrid = wx.FlexGridSizer(cols = 4)
        buttongrid.Add(self.radio1, -1, wx.EXPAND)
        buttongrid.Add((-1,-1))
        buttongrid.Add(button1, -1, wx.ALL, 10)
        buttongrid.Add(button2, -1, wx.ALL, 10)
        buttongrid.AddGrowableCol(1)
                
        boxer = wx.FlexGridSizer(cols=1)
        boxer.AddGrowableCol(0)
        boxer.AddGrowableRow(0)

        boxer.Add(self.list, 1, wx.EXPAND)
        boxer.Add(buttongrid, 1, wx.EXPAND | wx.ALL, 2)
        boxer.Add(self.check1, 1, wx.EXPAND| wx.ALL, 4)
        
        panel.SetSizer(boxer)
        panel.Layout()
        self.SetIcon(images.GetImage('rufus.ico'))
        
        wx.EVT_CLOSE(self, self.OnButtonCancel)
        wx.EVT_KEY_DOWN(panel, self.OnKeyDown)
        wx.EVT_KEY_DOWN(self.radio1, self.OnKeyDown)
        
        self.list.Populate(file_ranges)
        self.Show()
        
        
    def OnKeyDown(self, event):
        if event.GetKeyCode() == 27: #ESC key hit
            self.Cleanup()
        else:
            event.Skip()
    

    def OnRadio(self, event):
        if event.GetSelection() == 0:
            self.is_random = True
            self.toolbar.EnableTool(101, False)
            self.toolbar.EnableTool(102, False)
        else:
            self.is_random = False
            self.toolbar.EnableTool(101, True)
            self.toolbar.EnableTool(102, True)

    def OnRightClick(self, event):
        if not hasattr(self, "_popup_id"):
            self._popup_id = []
            for i in range(0, 5):
                self._popup_id.append(300 + i)
            
            wx.EVT_MENU(self, self._popup_id[0], self.SelectAllItems)
            wx.EVT_MENU(self, self._popup_id[1], self.InvSelection)
            wx.EVT_MENU(self, self._popup_id[2], self.SetCheckboxSelected)
            wx.EVT_MENU(self, self._popup_id[3], self.MoveToTop)
            wx.EVT_MENU(self, self._popup_id[4], self.MoveToBottom)

        menu = wx.Menu()

        menu.Append(self._popup_id[0], _("Select All"), _("Select All"))
        menu.Append(self._popup_id[1], _("Invert Selection"), _("Invert Selection"))
        menu.Append(self._popup_id[2], _("Check Checkbox(es)"), _("Check Checkbox(es)"))
        menu.AppendSeparator()
        menu.Append(self._popup_id[3], _("Move to top"), _("Move to top"))
        menu.Append(self._popup_id[4], _("Move to bottom"), _("Move to bottom"))

        self.PopupMenu(menu, self.list.ScreenToClient(wx.GetMousePosition()))
        menu.Destroy()
    
    def MoveToTop(self, event):
        idx = self.list.GetFirstSelected()
        while idx != -1:
            i = 0
            id = self.list.GetItemData(idx)
            for d in self.list.itemData:
                if d[0] == id:
                    item = self.list.itemData.pop(i)
                    self.list.itemData.insert(0, item)
                    break
                i += 1
            idx = self.list.GetNextSelected(idx)
        self.list.RePopulate()
    
    def MoveToBottom(self, event):
        idx = self.list.GetFirstSelected()
        while idx != -1:
            i = 0
            id = self.list.GetItemData(idx)
            for d in self.list.itemData:
                if d[0] == id:
                    item = self.list.itemData.pop(i)
                    self.list.itemData.append(item)
                    break
                i += 1
            idx = self.list.GetNextSelected(idx)
        self.list.RePopulate()
    
    
    def MoveUp(self, index, order):
        if index > order:
            item = self.list.itemData.pop(index)
            self.list.itemData.insert(index-1, item)


    def MoveDown(self, index, order):
        listSize = len(self.list.itemData)
        if index <  listSize - order - 1:
            item = self.list.itemData.pop(index)
            self.list.itemData.insert(index+1, item)
        
    def OnMoveUp(self, events):
        selected = []
        index = self.list.GetFirstSelected()
        while index != -1:
            selected.append(index)
            index = self.list.GetNextSelected(index)
        numberSelected=len(selected)
        for i in range(numberSelected):
            self.MoveUp(selected[i], i)
        self.list.RePopulate()

    
    def OnMoveDown(self, events):
        selected = []
        index = self.list.GetFirstSelected()
        while index != -1:
            selected.append(index)
            index = self.list.GetNextSelected(index)
        numberSelected = len(selected)
        for i in range(numberSelected):
            self.MoveDown(selected[numberSelected-i-1], i)
        self.list.RePopulate()

    def SelectAllItems(self, event):
        listSize = self.list.GetItemCount()
        for i in range(listSize):
            self.list.SetItemState(i, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
        self.list.RePopulate()

    def InvSelection(self, event):
        listSize = self.list.GetItemCount()
        for i in range(listSize):
            if not self.list.GetItemState(i, wx.LIST_STATE_SELECTED):
               self.list.SetItemState(i, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
            else: 
               self.list.SetItemState(i, 0, wx.LIST_STATE_SELECTED)
        self.list.RePopulate()

    def SetCheckboxSelected(self, event):
        listSize = self.list.GetItemCount()
        for i in range(listSize):
            if self.list.GetItemState(i, wx.LIST_STATE_SELECTED):
                key = self.list.GetItemData(i)
                self.list.OnEbedClick(event, key)
        self.list.RePopulate()

    def OnButtonOk(self, event):
        ranges = []
        
        for d in self.list.itemData:
            checked = self.list.ebed_ctrl_data[d[0]][0]
            ranges.append([d[4][0], d[4][1], d[4][2], d[4][3], checked])

        self.SetRanges(ranges, self.is_random)
        self.OnButtonCancel(event)
 
    
    def OnButtonCancel(self, event):
        self.btconfig['use_multimngr'] = self.check1.GetValue()
        self.Destroy()
            

if __name__ == "__main__":
    _ = lambda x: x # needed for gettext
    ranges = [((u'Temptation Waits.mp3', 7815029L), 0L, 29L, 0.0, True),
             ((u"I Think I', Paranoid.mp3", 5589934L), 29L, 51L, 0.0, True),
             ((u'When I Grow Up.mp3', 5555486L), 51L, 72L, 0.0, True),
             ((u'test.mp3', 6436230L), 72L, 96L, 0.0, True),
             ((u'test2.mp3', 6436230L), 72L, 96L, 0.0, True),
             ((u'test3.mp3', 6436230L), 72L, 96L, 0.0, True),
             ((u'Special.mp3', 6575675L), 96L, 121L, 0.0, True)] 

    app = wx.PySimpleApp()
    t = DLOrderMngr(None, BTConfig(), Images(), ranges, False, None)
    t.Show(True)
    app.MainLoop()
