# Written by Bram Cohen
# see LICENSE.txt for license information

try:
    from mhash import MHASH, MHASH_SHA1
    mhash_flag = True
except:
    from sha import sha
    mhash_flag = False

import os
from bisect import bisect_right

class Storage:
    def __init__(self, files, open, exists, getsize):
        # can raise IOError and ValueError
        self.files = files
        self.ranges = []
        total = 0l
        so_far = 0l
        for file, length in files:
            if length != 0:
                self.ranges.append((total, total + length, file))
                total += length
                if exists(file):
                    l = getsize(file)
                    if l > length:
                        l = length
                    so_far += l
            elif not exists(file):
                open(file, 'wb').close()
        self.begins = [i[0] for i in self.ranges]
        self.total_length = total
        self.handles = {}
        self.whandles = {}
        self.tops = {}
        for file, length in files:
            if exists(file):
                l = getsize(file)
                if l != length:
                    self.handles[file] = open(file, 'rb+')
                    self.whandles[file] = 1
                    if l > length:
                        self.handles[file].truncate(length)
                else:
                    self.handles[file] = open(file, 'rb')
                self.tops[file] = l
            else:
                self.handles[file] = open(file, 'wb+')
                self.whandles[file] = 1

    def was_preallocated(self, pos, length):
        for file, begin, end in self._intervals(pos, length):
            if self.tops.get(file, 0) < end:
                return False
        return True

    def set_readonly(self):
        # may raise IOError or OSError
        for file in self.whandles.keys():
            old = self.handles[file]
            old.flush()
            old.close()
            self.handles[file] = open(file, 'rb')

    def get_total_length(self):
        return self.total_length

    def _intervals(self, pos, amount):
        r = []
        stop = pos + amount
        p = bisect_right(self.begins, pos) - 1
        while p < len(self.ranges) and self.ranges[p][0] < stop:
            begin, end, file = self.ranges[p]
            r.append((file, max(pos, begin) - begin, min(end, stop) - begin))
            p += 1
        return r

    def read(self, pos, amount):
        r = []
        for file, pos, end in self._intervals(pos, amount):
            h = self.handles[file]
            h.seek(pos)
            r.append(h.read(end - pos))
        return ''.join(r)

    def write(self, pos, s):
        # might raise an IOError
        total = 0
        for file, begin, end in self._intervals(pos, len(s)):
            if not self.whandles.has_key(file):
                self.handles[file].close()
                self.handles[file] = open(file, 'rb+')
                self.whandles[file] = 1
            h = self.handles[file]
            h.seek(begin)
            h.write(s[total: total + end - begin])
            total += end - begin

    def close(self):
        for h in self.handles.values():
            h.close()
            
    def grow_all(self):
        for begin, end, file in self.ranges:
            l = end - begin
            if l > self.tops.get(file, 0):
                h = self.handles[file]
                h.seek(l-1)
                h.write(chr(0xFF))

    def GetDateSizeData(self):
        mod_data = []
        for a, b, filename in self.ranges:
            mod_data.append((str(os.path.getsize(filename)), str(os.path.getmtime(filename))))
        return mod_data


def lrange(a, b, c):
    r = []
    while a < b:
        r.append(a)
        a += c
    return r
