"""
SAM message buffer

Written by duck in 2004 and released into the public domain.
"""

import shlex, string
import StringIO

class SamBuffer:

    def __init__(self):
        self.buffer = ''
        self.cmd = ''
        self.kwargs = ''
        self.size = 0

    def add(self, data):
        """Add data to the buffer"""
        self.buffer += data

    def _hasLine(self):
        """Check if there is a a newline present"""
        posNewline = string.find(self.buffer, '\n')
        return posNewline != -1

    def _getLine(self):
        """Get everything until the newline"""
        posNewline = string.find(self.buffer, '\n')
        line = self.buffer[:posNewline]
        self.buffer = self.buffer[posNewline+1:]
        return line

    def _hasData(self, size):
        """See if there is enough data"""
        return len(self.buffer) >= size

    def _getData(self, size):
        """Get the data"""
        data = self.buffer[:size]
        self.buffer = self.buffer[size:]
        return data

    def getMessages(self):
        """Return all complete messages on the buffer"""
        messages = []
        while True:
            # read til the newline
            if not self.cmd and self._hasLine():
                line = self._getLine()
                (cmd, kwargs) = self._samdecode(line)
                if 'SIZE' in kwargs:
                    self.size = int(kwargs['SIZE'])
                    self.cmd = cmd
                    self.kwargs = kwargs
                else:
                    messages.append((cmd, kwargs, None))
            # read the remaining data
            elif self.size and self._hasData(self.size):
                data = self._getData(self.size)
                messages.append((self.cmd, self.kwargs, data))
                self.cmd = None
                self.kwargs = None
                self.size = 0
            # nothing left? return
            else:
                return messages

    def _samdecode(self, s):
        """Given a SAM command, returns (a, b), where a is the string at
        the beginning of the command, and b is a dictionary of name,
        value pairs for the command. Stolen from pysam"""
        (args, kwargs) = ([], {})
        for w in self.arg_tokenize(s):
            if '=' in w:
                kwargs[w.split('=')[0]] = w.split('=')[1]
            else:
                args += [w]
        return (' '.join(args), kwargs)

    def arg_tokenize (self, argstr, use_new=True):
        """fixing/kludging missing shlex.split in Python <2.3"""
        if hasattr(shlex, 'split') and use_new:
            toks = shlex.split(argstr)
        else:
            import shlex_split
            toks = shlex_split.shlex_split(argstr)
        return toks

def main():
    sb = SamBuffer()
    sb.add("HELLO VERSION MIN=1.0 MAX=1.0\n")
    sb.add("HELLO VERSION MIN=1.0 MAX=1.0 SIZE=3\n321")
    sb.add("HELLO VERSION MIN=1.0 MESSAGE=\"Hello there\" SIZE=3\n321")
    for m,k,e in sb.getMessages():
        print m,k,e
    
if __name__ == '__main__':
    main()
