/***************************************************************************
Name: CS Lite
Description: Control cookie permissions.
Author: Ron Beckman
Homepage: http://addons.mozilla.org

Copyright (C) 2007  Ron Beckman

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

Free Software Foundation, Inc.
51 Franklin Street
Fifth Floor
Boston, MA  02110-1301
USA
***************************************************************************/




/*last10HostsInitObserver() is used to prevent an overflow of notifications
 from being sent to the observer in rapid succession. this also allows hosts
 to be added to the last10Hosts array from cookie-changed and cookie-rejected
 notifications in cookiesafeObservers.js before re-populating the listbox.*/

function last10HostsInitObserver() {
	cookiesafeLast10.createListItems();
	csHostsObserver.init();
}

var csHostsPrefObserver = {

	gPrefs: null,

	init: function() {
		this.gPrefs = this.getGlobalPrefs();
		this.gPrefs.QueryInterface(Components.interfaces.nsIPrefBranch2);
		this.gPrefs.addObserver("", this, false);
	},

	uninit: function() {
		this.gPrefs.removeObserver("", this);
	},

	QueryInterface : function(aIID) {
		if (aIID.equals(Components.interfaces.nsISupports) ||
		    aIID.equals(Components.interfaces.nsIObserver))
			return this;
		throw Components.results.NS_NOINTERFACE;
	},

	getGlobalPrefs: function() {
		return Components.classes["@mozilla.org/preferences-service;1"].
		getService(Components.interfaces.nsIPrefService).
		getBranch("network.cookie.");
	},

	observe: function(aSubject, aTopic, aData) {
		//make sure that if this is a pref notification that it relates
		//to the cookieBehavior or lifetimePolicy prefs
		if (aTopic == 'nsPref:changed' &&
		    (aData == 'cookieBehavior' ||
		     aData == 'lifetimePolicy')) {
			cookiesafeLast10.createListItems();
		}
	}
};

var csHostsObserver = {

	gPrefs: null,

	init: function() {
		/*the cookiesafe-hosts-changed notification is only sent when
		a perm-changed overflow is triggered*/

		var os = this.getObserver();
		os.addObserver(this, 'cookiesafe-hosts-changed', false);
		os.addObserver(this, 'csperm-changed', false);
		os.addObserver(this, 'perm-changed', false);
		os.addObserver(this, 'cookie-changed', false);
		os.addObserver(this, 'cookie-rejected', false);
	},

	uninit: function() {
		var os = this.getObserver();
		os.removeObserver(this, 'cookiesafe-hosts-changed');
		os.removeObserver(this, 'csperm-changed');
		os.removeObserver(this, 'perm-changed');
		os.removeObserver(this, 'cookie-changed');
		os.removeObserver(this, 'cookie-rejected');
	},

	QueryInterface : function(aIID) {
		if (aIID.equals(Components.interfaces.nsISupports) ||
		    aIID.equals(Components.interfaces.nsIObserver))
			return this;
		throw Components.results.NS_NOINTERFACE;
	},

	getObserver: function() {
		return Components.classes["@mozilla.org/observer-service;1"].
		getService(Components.interfaces.nsIObserverService);
	},

	getCS: function() {
		return Components.classes['@mozilla.org/CookieSafe;1'].
		createInstance(Components.interfaces.nsICookieSafe);
	},

	getCSHiddenMenuItems: function() {
		return Components.classes['@mozilla.org/CSHiddenMenuItems;1'].
		createInstance(Components.interfaces.nsICSHiddenMenuItems);
	},

	getCSLast10Hosts: function() {
		return Components.classes['@mozilla.org/CSLast10Hosts;1'].
		getService(Components.interfaces.nsICSLast10Hosts);
	},

	getCSTempExceptions: function() {
		return Components.classes['@mozilla.org/CSTempExceptions;1'].
		getService(Components.interfaces.nsICSTempExceptions);
	},

	getGlobalPrefs: function() {
		return Components.classes["@mozilla.org/preferences-service;1"].
		getService(Components.interfaces.nsIPrefService).
		getBranch("network.cookie.");
	},

	observe: function(aSubject, aTopic, aData) {
		this.uninit();
		window.setTimeout(last10HostsInitObserver,1050);
	}
};

var cookiesafeLast10 = {

	TB2: false,

	getWin: function(name) {
		return Components.classes["@mozilla.org/appshell/window-mediator;1"].
		getService(Components.interfaces.nsIWindowMediator).
		getMostRecentWindow(name);
	},

	getPrefs: function() {
		return Components.classes["@mozilla.org/preferences-service;1"].
		getService(Components.interfaces.nsIPrefService).
		getBranch("cookiesafe.");
	},

	getGlobalPrefs: function() {
		return Components.classes["@mozilla.org/preferences-service;1"].
		getService(Components.interfaces.nsIPrefService).
		getBranch("network.cookie.");
	},

	getURI: function(url) {
		return Components.classes["@mozilla.org/network/io-service;1"].
		getService(Components.interfaces.nsIIOService).
		newURI(url,null,null);
	},

	getCS: function() {
		return Components.classes['@mozilla.org/CookieSafe;1'].
		createInstance(Components.interfaces.nsICookieSafe);
	},

	getCSHiddenMenuItems: function() {
		return Components.classes['@mozilla.org/CSHiddenMenuItems;1'].
		createInstance(Components.interfaces.nsICSHiddenMenuItems);
	},

	getCSLast10Hosts: function() {
		return Components.classes['@mozilla.org/CSLast10Hosts;1'].
		getService(Components.interfaces.nsICSLast10Hosts);
	},

	getCSTempExceptions: function() {
		return Components.classes['@mozilla.org/CSTempExceptions;1'].
		getService(Components.interfaces.nsICSTempExceptions);
	},

	getPermManager: function() {
		//check if browser is TB2
		if (this.TB2) {
			return Components.classes["@mozilla.org/CSPermManager;1"].
			getService(Components.interfaces.nsICSPermManager);
		} else {
			return Components.classes["@mozilla.org/permissionmanager;1"].
			getService(Components.interfaces.nsIPermissionManager);
		}
	},

	getStr: function(name) {
		return document.getElementById("cookiesafe-bundle").getString(name);
	},

	init: function() {
		//check if browser is TB2
		var brows = this.getAppInfo();
		var num = parseInt(brows.version);
		if (brows.name=='Thunderbird' && num==2) {
			this.TB2 = true;
		}

		csHostsPrefObserver.init();
		csHostsObserver.init();
		this.createListItems();
	},

	exit: function() {
		csHostsPrefObserver.uninit();
		csHostsObserver.uninit();
	},

	createListItems: function() {
		var listitem,listcell;
		var lastten = this.getCSLast10Hosts();
		var hosts = lastten.getLastTenHosts();
		var box = document.getElementById('hosts');

		while(box.firstChild) {
			box.removeChild(box.firstChild);
		}

		//make sure you remove all elements from listbox before returning
		if (!hosts) return false;

		hosts = hosts.split(' ');
		for (var i=0; i<hosts.length; ++i) {
			if (!hosts[i]) continue;
			listitem = document.createElement('listitem');
			listitem.setAttribute('value',hosts[i]);
			listcell = document.createElement('listcell');
			listcell.className = 'listcell-iconic ' + this.testException(hosts[i]);
			listcell.setAttribute('label',hosts[i]);
			listitem.appendChild(listcell);
			box.appendChild(listitem);
		}
		return false;
	},

	testException: function(host) {
		var img;
		var action = this.testPermission(host);

		if (action==8) {
			var tempExc = this.getCSTempExceptions();
			if (tempExc.testTempExceptions(host)) {
				img = 'cs-temporary';
			} else {
				img = 'cs-session';
			}
		} else if (action==1) {
			img = 'cs-allow';
		} else if (action==2) {
			img = 'cs-block';
		}

		if (!action) {
			var gPrefs = this.getGlobalPrefs();
			var behavior = gPrefs.getIntPref('cookieBehavior');
			var policy = gPrefs.getIntPref('lifetimePolicy');
			if (behavior==2) {
				img = 'cs-block';
			} else {
				img = (policy==2) ? 'cs-session' : 'cs-allow';
			}
		}
		return img;
	},

	testPermission: function(host) {
		var url = (host=='scheme:file') ? 'file:///cookiesafe' : 'http://'+host;
		var uri = this.getURI(url);
		var mngr = this.getPermManager();
		var action = mngr.testPermission(uri,'cookie');
		return action;
	},

	getAppInfo: function() {
		return Components.classes['@mozilla.org/xre/app-info;1'].
		createInstance(Components.interfaces.nsIXULAppInfo);
	},

	getWindowTypeForApp: function(name) {
		switch (name) {
			case 'Thunderbird':
				return 'mail:3pane';

			case 'Songbird':
				return 'Songbird:Main';

			case 'eMusic Remote':
				return 'emusic:window';

			default:
				return 'navigator:browser';
		}
	},

	addException: function(exc) {
		var errors=0;
		var brows = this.getAppInfo();
		var win = this.getWin(this.getWindowTypeForApp(brows.name));
		var box = document.getElementById('hosts');

		if (!box.selectedItems[0]) return false;

		/*uninit the hosts observer to avoid causing the listbox
		to update before all of the exceptions have been added*/
		csHostsObserver.uninit();

		for (var i=0; i<box.selectedItems.length; ++i) {
			try {
				win.cookiesafe.addException2(box.selectedItems[i].value,exc);
			} catch(e) {
				++errors;
				continue;
			}
		}

		//update the listbox
		this.createListItems();

		//re-initialize the hosts observer
		csHostsObserver.init();

		if (!errors) {
			//the success alert has been temporarily removed since the icons indicate
			//wether the permission was changed or not. since the user already has a
			//visual confirmation with the icons the alert box is rather unnecessary
			//alert(this.getStr('cookiesafe.AddExcSuccess'));
		} else {
			alert(this.getStr('cookiesafe.AddExcError'));
		}
		return false;
	}
};
