/***************************************************************************
Name: CookieSafe
Description: Control cookie permissions.
Author: Ron Beckman
Homepage: http://addons.mozilla.org

Copyright (C) 2007  Ron Beckman

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

Free Software Foundation, Inc.
51 Franklin Street
Fifth Floor
Boston, MA  02110-1301
USA
***************************************************************************/


var editCookiesObserver = {

	init: function() {
		var os = this.getObserver();
		os.addObserver(this, 'cookie-changed', false);
	},

	uninit: function() {
		var os = this.getObserver();
		os.removeObserver(this, 'cookie-changed');
	},

	QueryInterface : function(aIID) {
		if (aIID.equals(Components.interfaces.nsISupports) ||
		    aIID.equals(Components.interfaces.nsIObserver))
			return this;
		throw Components.results.NS_NOINTERFACE;
	},

	getObserver: function() {
		return Components.classes["@mozilla.org/observer-service;1"].
		getService(Components.interfaces.nsIObserverService);
	},

	observe: function(aSubject, aTopic, aData) {
		if (aTopic=='cookie-changed') {
			cookiesafeEditCookies.filterTree();
		}
	}
};


var cookiesafeEditCookies = {

	init: function() {
		//change background color in Songbird
		var brows = this.getAppInfo();
		if (brows.name == 'Songbird') {
			var cig = document.getElementById('cookieInfoGrid');
			cig.className = 'songbird-cookieInfoGrid';
		}

		//register cookie observer
		editCookiesObserver.init();

		//set the expires popup onload
		var dom = this.getDOM();
		var index = dom.expires.selectedIndex;
		if (index == -1 || index == 6) dom.expires.value = '';

		this.filterTree();

		//make sure all content is visible
		var vbox = document.getElementById('topVbox');
		var width = parseInt(vbox.boxObject.width);
		var height = parseInt(vbox.boxObject.height);
		if (parseInt(window.innerWidth) < width) window.innerWidth = width + 15;
		if (parseInt(window.innerHeight) < height) window.innerHeight = height + 15;
	},

	uninit: function() {
		//unregister cookie observer
		editCookiesObserver.uninit();
	},

	filterTree: function(filter) {
		//determine which column to sort by
		var tree = document.getElementById('cookies-tree');
		var sortBy = tree.getAttribute('lastClicked');
		if (!sortBy) {
			sortBy = 'host';
			tree.setAttribute('lastClicked',sortBy);
		}

		//save the currently selected row
		var selection = (tree.currentIndex==-1) ? tree.getAttribute('index') : tree.currentIndex;
		selection = parseInt(selection);

		//determine whether to sort a to z
		var col = document.getElementById(sortBy);
		var atoz = col.getAttribute('atoz');
		atoz = (!atoz || atoz=='true') ? true : false;
		col.setAttribute('atoz',atoz);

		//determine whether to filter tree for current site
		if (filter == null) {
			var menu = document.getElementById('cookies-filter');
			filter = (menu.selectedIndex <= 0) ? false : true;
		}

		this.populateTree(sortBy,atoz,filter);

		if (!isNaN(selection) && selection > -1) {
			tree.treeBoxObject.ensureRowIsVisible(selection);
		}
	},

	sortCol: function(sortBy) {
		//save the name of the column we're sorting
		var tree = document.getElementById('cookies-tree');
		var lastClicked = tree.getAttribute('lastClicked');

		//determine whether to sort a to z
		var col = document.getElementById(sortBy);
		var atoz = col.getAttribute('atoz');
		if (sortBy==lastClicked) {
			atoz = (atoz && atoz=='true') ? false : true;
		} else {
			atoz = (!atoz || atoz=='true') ? true : false;
		}
		col.setAttribute('atoz',atoz);

		//determine whether to filter tree for current site
		var menu = document.getElementById('cookies-filter');
		var filter = (menu.selectedIndex <= 0) ? false : true;

		tree.setAttribute('lastClicked',sortBy);

		this.populateTree(sortBy,atoz,filter);
	},

	hideTree: function(tf) {
		document.getElementById('cookiesTreeArea').setAttribute('collapsed',tf);
		if (!tf) this.filterTree();

		//make sure all content is visible
		var vbox = document.getElementById('topVbox');
		var width = parseInt(vbox.boxObject.width);
		var height = parseInt(vbox.boxObject.height);
		if (parseInt(window.innerWidth) < width) window.innerWidth = width + 15;
		if (parseInt(window.innerHeight) < height) window.innerHeight = height + 15;
	},

	getDOM: function() {
		var dom = {};
		dom.host = document.getElementById('cookies-txt-host');
		dom.path = document.getElementById('cookies-txt-path');
		dom.name = document.getElementById('cookies-txt-name');
		dom.value = document.getElementById('cookies-txt-value');
		dom.secure = document.getElementById('cookies-popup-secure');
		dom.expires = document.getElementById('cookies-popup-expires');
		return dom;
	},

	clearExpires: function() {
		var dom = this.getDOM();
		dom.expires.value = '';
	},

	fillTextBoxes: function() {
		var data = this.getData();
		if (data.length < 7) return false;

		var dom = this.getDOM();
		dom.host.value = data[0];
		dom.path.value = data[1];
		dom.name.value = data[2];
		dom.value.value = data[3];
		dom.secure.selectedIndex = (data[4]=='true') ? 1 : 0;
		dom.expires.selectedIndex = 6;
		dom.expires.value = data[6];

		return false;
	},

	resetTextBoxes: function() {
		var dom = this.getDOM();

		//save the currently selected expires item
		var exp = dom.expires.selectedIndex;

		//clear tree selections
		var tree = document.getElementById('cookies-tree');
		if (tree.view && tree.view.selection) tree.view.selection.clearSelection();

		dom.host.value = '';
		dom.path.value = '/';
		dom.name.value = '';
		dom.value.value = '';
		dom.secure.selectedIndex = 0;
		dom.expires.selectedIndex = exp;
		if (exp == -1 || exp == 6) dom.expires.value = '';
	},

	getData: function() {
		var item = [];
		var tree = document.getElementById('cookies-tree');

		try {
			for (var i=0; i<tree.columns.count; ++i) {
				item.push(tree.view.getCellText(tree.currentIndex,tree.columns.getColumnAt(i)));
			}
		} catch(e) {
			return [];
		}

		return item;
	},

	populateTree: function(sortBy,atoz,filter) {
		var cookies = (!filter) ? this.getAllCookies(sortBy) : this.getCurrentCookies(sortBy);

		if (!atoz) cookies.reverse();

		//clear text boxes and tree selection
		this.resetTextBoxes();

		//remove all tree items
		var child = document.getElementById('cookies-child');
		while (child.firstChild) {
			child.removeChild(child.firstChild);
		}

		for (var i=0; i<cookies.length; ++i) {
			child.appendChild(this.createItem(cookies[i]));
		}
	},

	createItem: function(line) {
		var item = document.createElement('treeitem');
		var row = document.createElement('treerow');

		for (var i=1; i<line.length; ++i) {
			var cell = document.createElement('treecell');
			cell.setAttribute('label',line[i]);
			row.appendChild(cell);
		}

		item.appendChild(row);
		return item;
	},

	getAllCookies: function(sortBy) {
		var exp,dt,ck;
		var cookies = [];
		var mngr = this.getCookieManager();
		var cks = mngr.enumerator;
		while (('hasMoreElements' in cks && cks.hasMoreElements()) ||
			 ('hasMore' in cks && cks.hasMore())) {
			ck = cks.getNext();
			ck.QueryInterface(Components.interfaces.nsICookie);
			//set expiration label in xx/xx/xx format
			if (ck.expires==0) {
				exp = this.getStr('cookiesafe.lSession');
			} else {
				dt = new Date();
				dt.setTime(ck.expires * 1000);
				exp = (dt.getMonth()+1)+'/'+dt.getDate()+'/'+dt.getFullYear();
			}
			cookies.push([ck[sortBy],ck.host,ck.path,ck.name,ck.value,ck.isSecure,exp,ck.expires]);
		}
		cookies.sort();
		return cookies;
	},

	getWindowTypeForApp: function(name) {
		switch (name) {
			case 'Thunderbird':
				return 'mail:3pane';

			case 'Songbird':
				return 'Songbird:Main';

			case 'eMusic Remote':
				return 'emusic:window';

			default:
				return 'navigator:browser';
		}
	},

	getCurrentCookies: function(sortBy) {
		var exp,dt,ck;
		var cookies = [];
		var cs = this.getCS();
		var brows = this.getAppInfo();
		var win = this.getWin(this.getWindowTypeForApp(brows.name));
		var site = win.cookiesafe.getDomain();
		var base = cs.removeSub(site);
		var mngr = this.getCookieManager();
		var cks = mngr.enumerator;
		while (('hasMoreElements' in cks && cks.hasMoreElements()) ||
			 ('hasMore' in cks && cks.hasMore())) {
			ck = cks.getNext();
			ck.QueryInterface(Components.interfaces.nsICookie);
			ck.QueryInterface(Components.interfaces.nsICookie2);
			//set expiration label in xx/xx/xx format
			if (ck.expires==0) {
				exp = this.getStr('cookiesafe.lSession');
			} else {
				dt = new Date();
				dt.setTime(ck.expires * 1000);
				exp = (dt.getMonth()+1)+'/'+dt.getDate()+'/'+dt.getFullYear();
			}
			if (ck.rawHost==base || ck.host.indexOf('.'+base)!=-1) {
				cookies.push([ck[sortBy],ck.host,ck.path,ck.name,ck.value,ck.isSecure,exp,ck.expires]);
			}
		}
		cookies.sort();
		return cookies;
	},

	getCS: function() {
		return Components.classes['@mozilla.org/CookieSafe;1'].
		createInstance(Components.interfaces.nsICookieSafe);
	},

	getCSHiddenMenuItems: function() {
		return Components.classes['@mozilla.org/CSHiddenMenuItems;1'].
		createInstance(Components.interfaces.nsICSHiddenMenuItems);
	},

	getCSLast10Hosts: function() {
		return Components.classes['@mozilla.org/CSLast10Hosts;1'].
		getService(Components.interfaces.nsICSLast10Hosts);
	},

	getCSTempExceptions: function() {
		return Components.classes['@mozilla.org/CSTempExceptions;1'].
		getService(Components.interfaces.nsICSTempExceptions);
	},

	getWin: function(name) {
		return Components.classes["@mozilla.org/appshell/window-mediator;1"].
		getService(Components.interfaces.nsIWindowMediator).
		getMostRecentWindow(name);
	},

	getAppInfo: function() {
		return Components.classes['@mozilla.org/xre/app-info;1'].
		createInstance(Components.interfaces.nsIXULAppInfo);
	},

	getStr: function(name) {
		return document.getElementById("cookiesafe-bundle").getString(name);
	},

	getCookieManager: function() {
		return Components.classes["@mozilla.org/cookiemanager;1"].
		getService(Components.interfaces.nsICookieManager);
	},

	getCookieManager2: function() {
		return Components.classes["@mozilla.org/cookiemanager;1"].
		getService(Components.interfaces.nsICookieManager2);
	},

	isValid: function(item) {
		if (item.id=='cookies-txt-host') {
			//allow periods, dashes
			item.value = item.value.replace(/[~`!@#$%^&*()_+|=\\{}\[\]:\";\'<>?,\/ ]/g,'');
		} else if (item.id=='cookies-txt-path') {
			//allow forward slashes, dashes, underscores
			item.value = item.value.replace(/[~`!@#$%^&*()+|=\\{}\[\]:\";\'<>?,. ]/g,'');
		}
		//do nothing for cookies-txt-name, cookies-txt-value and cookies-popup-expires
	},

	create: function() {
		var dom = this.getDOM();
		if (!dom.name.value || !dom.value.value) return false;

		var expires;
		var dt = new Date();
		var time = parseInt(dt.getTime() / 1000);
		if (!dom.path.value) dom.path.value = '/';

		var index = dom.expires.selectedIndex;
		if (index == -1 || index == 6) {
			expires = parseInt(dom.expires.value.replace(/\D/g,''));
		} else {
			expires = parseInt(dom.expires.selectedItem.value);
		}

		if (isNaN(expires) || !expires || (index < 0 && expires <= time)) expires = 0;
		var session = (index==0 || !expires) ? true : false;
		var secure = (dom.secure.value=='true') ? true : false;

		if (session) {
			expires = time + 86400;
		} else if (index > 0 && index < 6) {
			expires += time;
		}

		try {
			var mngr = this.getCookieManager2();
			if ('cookieExists' in mngr) {
				mngr.add(dom.host.value,dom.path.value,dom.name.value,dom.value.value,secure,true,session,expires);
			} else {
				mngr.add(dom.host.value,dom.path.value,dom.name.value,dom.value.value,secure,session,expires);
			}
			alert(this.getStr('cookiesafe.AddCookieSuccess'));
		} catch(e) {
			alert(this.getStr('cookiesafe.AddCookieError'));
		}
		return false;
	}
};
