/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2007 Phex Development Group
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  --- SVN Information ---
 *  $Id: OIOServer.java 4140 2008-03-03 00:33:07Z complication $
 */
package phex.net;

import java.io.IOException;
import java.net.BindException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.util.TimerTask;

import phex.common.Environment;
import phex.common.ThreadPool;
import phex.common.address.DefaultDestAddress;
import phex.common.address.DestAddress;
import phex.common.address.IpAddress;
import phex.common.log.NLogger;
import phex.connection.IncomingConnectionDispatcher;
import phex.host.HostManager;
import phex.host.NetworkHostsContainer;
import phex.net.repres.PresentationManager;
import phex.net.repres.SocketFacade;
import phex.net.repres.def.DefaultSocketFacade;
import phex.prefs.core.NetworkPrefs;
import phex.security.AccessType;
import phex.security.PhexSecurityException;
import phex.security.PhexSecurityManager;
import phex.servent.Servent;

public class OIOServer extends Server
{
    protected ServerSocket serverSocket;
    
    public OIOServer()
    {
        super();
        Environment.getInstance().scheduleTimerTask( 
            new FirewallCheckTimer(), FirewallCheckTimer.TIMER_PERIOD,
            FirewallCheckTimer.TIMER_PERIOD );
    }

    public int getListeningLocalPort()
    {
        if ( serverSocket != null )
        {
            return serverSocket.getLocalPort();
        }
        else
        {
            return NetworkPrefs.ListeningPort.get().intValue();
        }
    }
    
    public IpAddress resolveLocalHostIP()
    {
        byte[] ip = null;
        InetAddress addr = serverSocket.getInetAddress();
        ip = addr.getAddress();
        if (ip[0] == 0 && ip[1] == 0 && ip[2] == 0 && ip[3] == 0)
        {
            ip = IpAddress.LOCAL_HOST_IP;
        }
        IpAddress ipAddress = new IpAddress( ip );
        return ipAddress;
    }

    // The listening thread.
    public void run()
    {
        if (NLogger.isDebugEnabled( OIOServer.class ))
            NLogger.debug( OIOServer.class,
                "Listener started. Listening on: "
                    + serverSocket.getInetAddress().getHostAddress() + ':'
                    + serverSocket.getLocalPort());
        
        try
        {
            while ( serverSocket != null && !serverSocket.isClosed() )
            {  
                try
                {
                    Socket incoming = serverSocket.accept();
                    // create facade...
                    DefaultSocketFacade incomingFacade = new DefaultSocketFacade( 
                        incoming );
                    handleIncomingSocket( incomingFacade );
                }
                catch ( SocketException exp )
                {
                    NLogger.debug( OIOServer.class, exp );
                }
                catch ( PhexSecurityException exp )
                {
                    NLogger.debug( OIOServer.class, exp );
                }
                catch (IOException exp)
                {
                    NLogger.error( OIOServer.class, exp, exp);
                }
            }
        }
        catch ( Exception exp )
        {
            NLogger.error( OIOServer.class, exp, exp );
        }

        isRunning = false;
        NLogger.debug( OIOServer.class, "Listener stopped.");
        DestAddress newAddress = PresentationManager.getInstance().createHostAddress(
            new IpAddress(IpAddress.LOCAL_HOST_IP), DefaultDestAddress.DEFAULT_PORT );
        localAddress.updateLocalAddress( newAddress );
        synchronized(this)
        {
            notifyAll();
        }
    }

    /**
     * @param socketChannel
     * @throws PhexSecurityException
     * @throws SocketException
     */
    private void handleIncomingSocket(SocketFacade clientSocket )
        throws IOException, PhexSecurityException
    {        
        clientSocket.setSoTimeout( NetworkPrefs.TcpRWTimeout.get().intValue() );

        DestAddress address = clientSocket.getRemoteAddress();
        NetworkHostsContainer netHostsContainer = Servent.getInstance().getHostService()
            .getNetworkHostsContainer();

        // if not already connected and connection is not from a private address.
        IpAddress remoteIp = address.getIpAddress();
        assert remoteIp != null;
        if (!netHostsContainer.isConnectedToHost(address)
            && !remoteIp.isSiteLocalIP() )
        {
            hasConnectedIncomming = true;
            lastInConnectionTime = System.currentTimeMillis();
        }
        
        // Set this will defeat the Nagle Algorithm, making short bursts of
        // transmission faster, but will be worse for the overall network.
        // incoming.setTcpNoDelay(true);

        // Create a Host object for the incoming connection
        // and hand it off to a ReadWorker to handle.
        AccessType access = PhexSecurityManager.getInstance()
            .controlHostAddressAccess(address);
        switch (access)
        {
            case ACCESS_DENIED:
            case ACCESS_STRONGLY_DENIED:
                throw new PhexSecurityException("Host access denied: " + address );
        }

        NLogger.debug( OIOServer.class, 
            "Accepted incoming connection from: "
                + address.getFullHostName());

        IncomingConnectionDispatcher dispatcher = new IncomingConnectionDispatcher(
            clientSocket, Servent.getInstance() );
        ThreadPool.getInstance().addJob( dispatcher,
            "IncomingConnectionDispatcher-" + Integer.toHexString(hashCode()));
    }

    @Override
    protected synchronized void bind( int initialPort ) throws IOException
    {
        assert (serverSocket == null);

        serverSocket = new ServerSocket();

        // Create a listening socket at the port.
        int tries = 0;
        boolean error;
        // try to find new port if port not valid
        do
        {
            error = false;

            try
            {
                NLogger.debug( OIOServer.class, "Binding to port " + initialPort );
                serverSocket.bind(new InetSocketAddress( initialPort ));
            }
            catch (BindException exp)
            {
                NLogger.debug( OIOServer.class, "Binding failed to port " + initialPort );
                if (tries > 50)
                {
                    throw exp;
                }
                error = true;
                initialPort++;
                tries++;
            }
        }
        while (error == true);

        IpAddress hostIP = resolveLocalHostIP();
        initialPort = serverSocket.getLocalPort();
        DestAddress newAddress = PresentationManager.getInstance().createHostAddress(
            hostIP, initialPort );
        localAddress.updateLocalAddress( newAddress );
    }
    
    @Override
    protected synchronized void closeServer()
    {
        if ( serverSocket != null )
        {
            try
            {
                serverSocket.close();
            }
            catch (IOException exp)
            {// ignore
            }
            serverSocket = null;
        }
    }
}