/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2006 Phex Development Group
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  --- CVS Information ---
 *  $Id: BandwidthManager.java 3832 2007-06-24 22:03:00Z gregork $
 */
package phex.common.bandwidth;

import phex.common.*;
import phex.prefs.core.BandwidthPrefs;

/**
 * A Manager class that manages all bandwidth controllers
 * and allows fast access to the controllers.
 */
public class BandwidthManager extends AbstractManager
{
    private BandwidthController phexBandwidthController;
    private BandwidthController networkBandwidthController;
    private BandwidthController downloadBandwidthController;
    private BandwidthController uploadBandwidthController;

    public void setDownloadBandwidth( int newDownloadBwInBytes )
    {
        BandwidthPrefs.MaxDownloadBandwidth.set( new Integer( newDownloadBwInBytes ) );
        downloadBandwidthController.setThrottlingRate(newDownloadBwInBytes);
    }

    public void setNetworkBandwidth(int newNetworkBwInBytes)
    {
        BandwidthPrefs.MaxNetworkBandwidth.set( new Integer( newNetworkBwInBytes ) );
        networkBandwidthController.setThrottlingRate(newNetworkBwInBytes);
    }
    
    public void setPhexTotalBandwidth(int newPhexBwInBytes)
    {
        BandwidthPrefs.MaxTotalBandwidth.set( new Integer( newPhexBwInBytes ) );
        phexBandwidthController.setThrottlingRate(newPhexBwInBytes);
    }

    public void setUploadBandwidth(int newUploadBwInBytes)
    {
        BandwidthPrefs.MaxUploadBandwidth.set( new Integer( newUploadBwInBytes ) );
        uploadBandwidthController.setThrottlingRate(newUploadBwInBytes);
    }

    public BandwidthController getPhexBandwidthController()
    {
        return phexBandwidthController;
    }

    public BandwidthController getNetworkBandwidthController()
    {
        return networkBandwidthController;
    }

    public BandwidthController getDownloadBandwidthController()
    {
        return downloadBandwidthController;
    }

    public BandwidthController getUploadBandwidthController()
    {
        return uploadBandwidthController;
    }
    
    //////////////////////// Start Manager interface ///////////////////////////

    static private class Holder
    {
       static protected final BandwidthManager manager = new BandwidthManager();
    }

    static public BandwidthManager getInstance()
    {
        return BandwidthManager.Holder.manager;
    }


    /**
     * This method is called in order to initialize the manager. This method
     * includes all tasks that must be done to intialize all the several manager.
     * Like instantiating the singleton instance of the manager. Inside
     * this method you can't rely on the availability of other managers.
     * @return true is initialization was successful, false otherwise.
     */
    public boolean initialize()
    {   
        phexBandwidthController =
            BandwidthController.acquireBandwidthController("PhexThrottle",
                BandwidthPrefs.MaxTotalBandwidth.get().intValue() );
        phexBandwidthController.activateShortTransferAvg(1000, 5);
        phexBandwidthController.activateLongTransferAvg(2000, 90);
            
        networkBandwidthController =
            BandwidthController.acquireBandwidthController("NetworkThrottle",
                BandwidthPrefs.MaxNetworkBandwidth.get().intValue() );
        networkBandwidthController.activateShortTransferAvg(1000, 5);
        networkBandwidthController.activateLongTransferAvg(2000, 90);
        networkBandwidthController.linkControllerIntoChain(
            phexBandwidthController);

        downloadBandwidthController =
            BandwidthController.acquireBandwidthController("DownloadThrottle",
                BandwidthPrefs.MaxDownloadBandwidth.get().intValue() );
        downloadBandwidthController.activateShortTransferAvg(1000, 5);
        downloadBandwidthController.activateLongTransferAvg(2000, 90);
        downloadBandwidthController.linkControllerIntoChain(
            phexBandwidthController);

        uploadBandwidthController =
            BandwidthController.acquireBandwidthController("UploadThrottle", 
                BandwidthPrefs.MaxUploadBandwidth.get().intValue() );
        uploadBandwidthController.activateShortTransferAvg(1000, 5);
        uploadBandwidthController.activateLongTransferAvg(2000, 90);
        uploadBandwidthController.linkControllerIntoChain(
            phexBandwidthController);
        return true;
    }

    /**
     * This method is called in order to perform post initialization of the
     * manager. This method includes all tasks that must be done after initializing
     * all the several managers. Inside this method you can rely on the
     * availability of other managers.
     * @return true is initialization was successful, false otherwise.
     */
    public boolean onPostInitialization()
    {
        return true;
    }
    
    /**
     * This method is called after the complete application including GUI completed
     * its startup process. This notification must be used to activate runtime
     * processes that needs to be performed once the application has successfully
     * completed startup.
     */
    public void startupCompletedNotify()
    {
    }

    /**
     * This method is called in order to cleanly shutdown the manager. It
     * should contain all cleanup operations to ensure a nice shutdown of Phex.
     */
    public void shutdown()
    {
        BandwidthController.releaseController( downloadBandwidthController );
        BandwidthController.releaseController( uploadBandwidthController );
        BandwidthController.releaseController( networkBandwidthController );
        BandwidthController.releaseController( phexBandwidthController );
    }
    //////////////////////// End Manager interface ///////////////////////////
}
