/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2007 Phex Development Group
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Created on 03.12.2005
 *  --- CVS Information ---
 *  $Id: ManagerController.java 4140 2008-03-03 00:33:07Z complication $
 */
package phex.common;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import phex.common.log.NLogger;
import phex.servent.Servent;
import phex.servent.OnlineStatus;
import phex.prefs.core.ConnectionPrefs;

public final class ManagerController
{
    static private final List<Manager> MANAGER_LIST;
    static
    {
        // I2PFIXME: IMPORTANT:
        // Below I attempt to crudely disable some components.
        // However, skipping getInstance() calls here may not suffice,
        // as unforeseen other components could execute those calls later.
        // Thus it seems safest that getInstace() methods
        // of all undesired Manager classes should reliably return null.
        
        List<Manager> tmpList = new ArrayList<Manager>();
        tmpList.add( phex.download.swarming.SwarmingManager.getInstance() );
        // I2PMOD: also instantiate a PresentationManager.
        tmpList.add( phex.net.repres.PresentationManager.getInstance() );
        tmpList.add( phex.security.PhexSecurityManager.getInstance() );
        tmpList.add( phex.statistic.StatisticsManager.getInstance() );
        tmpList.add( phex.common.bandwidth.BandwidthManager.getInstance() );
        MANAGER_LIST = Collections.unmodifiableList( tmpList );

        // Managers created, try bringing Servent into initial online status.
        if ( ConnectionPrefs.AutoConnectOnStartup.get().booleanValue() )
        {
            Servent.getInstance().setOnlineStatus( OnlineStatus.ONLINE );
        }
        else
        {
            Servent.getInstance().setOnlineStatus( OnlineStatus.OFFLINE );
        }
    }
    
    public static void initializeManagers()
    {
        for( Manager manager : MANAGER_LIST )
        {
            long start = System.currentTimeMillis();
            NLogger.debug( ManagerController.class,
                "Initializing " + manager.getClass().getName() );
            boolean succ = manager.initialize();
            if ( !succ )
            {
                NLogger.error( ManagerController.class,
                    "Failed to initialize " + manager.getClass().getName() );
                throw new RuntimeException( "Failed to initialize " +
                    manager.getClass().getName() );
            }
            long end = System.currentTimeMillis();
            NLogger.debug( ManagerController.class, 
                "Initialization time: " + (end-start) + " - " 
                + manager.getClass().getName());
        }
        
        postInitializeManagers();
    }

    /**
     * 
     */
    private static void postInitializeManagers()
    {
        for( Manager manager : MANAGER_LIST )
        {
            long start = System.currentTimeMillis();
            NLogger.debug( ManagerController.class,
                "On post initialization " + manager.getClass().getName() );
            boolean succ = manager.onPostInitialization();
            if ( !succ )
            {
                NLogger.error( ManagerController.class,
                    "Failed to initialize " + manager.getClass().getName() );
                throw new RuntimeException( "Failed to initialize " +
                    manager.getClass().getName() );
            }
            long end = System.currentTimeMillis();
            NLogger.debug( ManagerController.class, 
                "Post-Initialization time: " + (end-start) + " - " 
                + manager.getClass().getName());
        }
    }
    
    /**
     * This method is called after the complete application including GUI completed
     * its startup process.
     */
    public static void startupCompletedNotify()
    {
        for( Manager manager : MANAGER_LIST )
        {
            long start = System.currentTimeMillis();
            NLogger.debug( ManagerController.class,
                "StartupCompletedNotify " + manager.getClass().getName() );
            manager.startupCompletedNotify();
            long end = System.currentTimeMillis();
            NLogger.debug( ManagerController.class, 
                "StartupCompletedNotify time: " + (end-start) + " - " 
                + manager.getClass().getName());
        }
    }    

    public static void shutdownManagers( )
    {
        for( Manager manager : MANAGER_LIST )
        {
            try
            {
                long start = System.currentTimeMillis();
                NLogger.debug( ManagerController.class,
                    "Shutdown " + manager.getClass().getName() );
                manager.shutdown();
                long end = System.currentTimeMillis();
                NLogger.debug( ManagerController.class, 
                    "Shutdown time: " + (end-start) + " - " 
                    + manager.getClass().getName());
            }
            catch ( Exception exp )
            {
                NLogger.error( ManagerController.class, exp, exp  );
            }
        }
    }
}