/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2007 Phex Development Group
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  --- SVN Information ---
 *  $Id: ChatEngine.java 4153 2008-03-18 07:35:17Z ArneBab $
 */
package phex.chat;

import java.io.IOException;

import phex.common.Environment;
import phex.common.ThreadPool;
import phex.common.address.DestAddress;
import phex.common.bandwidth.BandwidthController;
import phex.common.bandwidth.BandwidthManager;
import phex.common.log.NLogger;
import phex.event.PhexEventTopics;
import phex.io.buffer.ByteBuffer;
import phex.net.connection.Connection;
import phex.net.connection.SocketFactory;
import phex.net.repres.SocketFacade;
import phex.prefs.core.NetworkPrefs;
import phex.servent.Servent;
import phex.utils.GnutellaInputStream;

/**
 *
 */
public final class ChatEngine
{
    private final ChatService chatMgr;
    private final boolean isOutgoingConnection;
    
    private GnutellaInputStream chatReader;
    private Connection connection;
    private String chatNick;

    /**
     * The host address of the chat connection.
     */
    private final DestAddress hostAddress;

    /**
     * For incoming chat requests
     */
    protected ChatEngine( ChatService chatMgr, Connection connection )
        throws IOException
    {
        this.chatMgr = chatMgr;
        this.connection = connection;
        hostAddress = connection.getSocket().getRemoteAddress();
        chatReader = connection.getInputStream();

        chatNick = hostAddress.getFullHostName();
        finalizeHandshake();
        isOutgoingConnection = false;
    }

    /**
     * For outgoing chat requests. We need to connect to the host address first.
     */
    protected ChatEngine( ChatService chatMgr, DestAddress aHostAddress )
    {
        this.chatMgr = chatMgr;
        hostAddress = aHostAddress;
        isOutgoingConnection = true;
        chatNick = hostAddress.getFullHostName();
    }

    public void startChat()
    {
        ChatReadWorker worker = new ChatReadWorker();
        ThreadPool.getInstance().addJob( worker,
            "ChatReadWorker-" + Integer.toHexString(worker.hashCode()));
    }

    public void stopChat()
    {
        if (connection != null) {
            connection.disconnect();
            connection = null;
        }
        chatMgr.chatClosed( this );
    }

    public boolean isConnected()
    {
        return connection != null;
    }

    /**
     * Returns the host address the engine is connected to
     */
    public DestAddress getHostAddress()
    {
        return hostAddress;
    }

    public String getChatNick()
    {
        return chatNick;
    }

    /**
     * Sends a chat message to the connected servent
     */
    public void sendChatMessage( String message )
    {
        if ( connection == null )
        {
            chatMgr.fireChatConnectionFailed( ChatEngine.this );
        }
        try
        {
            connection.write( ByteBuffer.wrap(  (message + "\n").getBytes() ) );
        }
        catch ( IOException exp )
        {
            NLogger.warn( ChatEngine.class, exp, exp );
            stopChat();
        }
    }

    private void finalizeHandshake()
        throws IOException
    {
        connection.getSocket().setSoTimeout( NetworkPrefs.TcpRWTimeout.get().intValue() );

        // read the header that have been left in the stream after accepting the
        // connection.
        String line;
        String upLine;
        do
        {
            line = chatReader.readLine();
            NLogger.debug( ChatEngine.class, "Read Chat header: " + line );
            if ( line == null )
            {
                throw new IOException( "No handshake response from chat partner." );
            }
            upLine = line.toUpperCase();
            if ( upLine.startsWith( "X-NICKNAME:" ) )
            {

                chatNick = line.substring(11).trim();
                NLogger.debug( ChatEngine.class, "Chat Nick: " + chatNick );
            }
        }
        while ( line.length() > 0 );

        // we respond with "CHAT/0.1 200 OK\r\n\r\n" to finish the handshake.
        NLogger.debug( ChatEngine.class, "Sending: CHAT/0.1 200 OK");
        connection.write( ByteBuffer.wrap( ("CHAT/0.1 200 OK\r\n" +
            "User-Agent: " + Environment.getPhexVendor() + "\r\n" +
            //"X-Nickname: " + StrUtil.getAppNameVersion() + "\r\n" +
            "\r\n").getBytes() ) );

        // assume we read the final "CHAT/0.1 200 OK" followed by possible headers.
        do
        {
            line = chatReader.readLine();
            NLogger.debug( ChatEngine.class, "Read Chat response: " + line );
            if ( line == null )
            {
                throw new IOException( "No handshake response from chat partner." );
            }
        }
        while ( line.length() > 0 );

        connection.getSocket().setSoTimeout( 0 );
        // chat connection open notification will be fired through the chat manager
    }

    private void connectOutgoingChat()
        throws IOException
    {
        NLogger.debug( ChatEngine.class, "Connect outgoing to: " + hostAddress );
        SocketFacade socket = SocketFactory.connect( hostAddress );
        
        BandwidthController bwController = BandwidthManager.getInstance()
            .getNetworkBandwidthController();
        connection = new Connection(socket, bwController);

        // initialize the chat connection handshake
        // First send "CHAT CONNECT/0.1\r\n" and header data
        // only header currently is user agent.
        String message = "CHAT CONNECT/0.1\r\n" +
            "User-Agent: " + Environment.getPhexVendor() + "\r\n" +
            //"X-Nickname: " + StrUtil.getAppNameVersion() + "\r\n" +
            "\r\n";
        NLogger.debug( ChatEngine.class, "Sending: " + message );
        connection.write( ByteBuffer.wrap( message.getBytes() ) );

        chatReader = connection.getInputStream();

        // assume we read "CHAT/0.1 200 OK" and header data
        // TODO: check string and status, don't check for version since it might
        //       change only for "CHAT" and "200"
        String line;
        String upLine;
        do
        {
            line = chatReader.readLine();
            NLogger.debug( ChatEngine.class, "Read Chat header: " + line );
            if ( line == null )
            {
                throw new IOException( "No handshake response from chat partner." );
            }
            upLine = line.toUpperCase();
            if ( upLine.startsWith( "X-NICKNAME:" ) )
            {
                chatNick = line.substring(11).trim();
            }
        }
        while ( line.length() > 0 );

        // we respond with "CHAT/0.1 200 OK\r\n\r\n" to finish the handshake.
        connection.write( ByteBuffer.wrap( ("CHAT/0.1 200 OK\r\n" +
            "User-Agent: " + Environment.getPhexVendor() + "\r\n\r\n" ).getBytes() ) );
        socket.setSoTimeout( 0 );
        // chat connection open notification will be fired through the chat manager
    }

    /**
     * Hide the reading thread from the engine implementation.
     */
    private class ChatReadWorker implements Runnable
    {
        public void run()
        {
            if ( isOutgoingConnection )
            {
                try
                {
                    connectOutgoingChat();
                }
                catch ( IOException exp )
                {
                    stopChat();
                    return;
                }
            }
            String str;
            while ( true )
            {
                try
                {
                    str = chatReader.readLine();
                    if ( str == null )
                    {
                        throw new IOException( "Remote host diconnected chat." );
                    }
                    if ( str.length() == 0 )
                    {
                        continue;
                    }
                    NLogger.debug( ChatEngine.class, "Reading chat message: " + str );

                    Servent.getInstance().getEventService().publish( 
                        PhexEventTopics.Chat_Update,
                        new ChatEvent( ChatEvent.Type.MSG_REC, ChatEngine.this, str ) );
                }
                catch ( IOException exp )
                {
                    NLogger.debug( ChatEngine.class, exp, exp);
                    stopChat();
                    break;
                }
            }
        }
    }
}