/**
 * Copyright (C) 2009  HungryHobo@mail.i2p
 * 
 * The GPG fingerprint for HungryHobo@mail.i2p is:
 * 6DD3 EAA2 9990 29BC 4AD2 7486 1E2C 7B61 76DC DC12
 * 
 * This file is part of I2P-Bote.
 * I2P-Bote is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * I2P-Bote is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with I2P-Bote.  If not, see <http://www.gnu.org/licenses/>.
 */

package i2p.bote.network.kademlia;

import static i2p.bote.network.kademlia.KademliaConstants.K;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.i2p.data.Destination;

import org.junit.Before;
import org.junit.Test;

public class KBucketTest {
    private List<KBucket> buckets;
    private Map<KBucket, KademliaPeer[]> destinationMap;

    @Before
    public void setUp() throws Exception {
        assertTrue("This test only works if k<=20", K<=20);
        
        int[] depths = new int[] {0, 1, 5};
        buckets = new ArrayList<KBucket>();
        destinationMap = new HashMap<KBucket, KademliaPeer[]>();
        for (int depth: depths) {
            BigInteger startId, endId;
            KBucket bucket;
            
            KademliaPeer[] peers1 = new KademliaPeer[21];
            peers1[0] = new KademliaPeer(new Destination("c41n4lNINwohF7b0Q9Lk~XG1hKdmtLVnvsuYZ~Cp~oqdid6X80bje7mAmvbPIxrmpt~cP2YBhJdPK58gFBLyTGZ~aLrZ7~os6khqxFsdVtZpoGC4MrO5vtm0GolLicZT0GYX~rOjjLR9L~aPHHYD2TmSjo10lCf7PLCkdRDT~qDUfB~djfBIhoqSUQAUNu8GRbHNC9p6lshvH--do9Ja9bxVHrSs~Gesg4c6ylAwR8Y-w699LwL0Nh9vZnoLjIdCAckFK0NYpq4btc4zq4zQU4HWLPNSJVz1XzT4yfLoKt5ggYNDfzrKqbY5lcDmuKSSV-XyW4uX4MieqqUMInu40l~OiGs670zCIl-2X2EP-guPrgXUK0D1i7V3aeYpH8G~9Vhx8LmlBD5XPvKjr1EVjvbvm1scwLhg6S1pDiaCUkK~WtilESVa8lFFPshIlrFsX0jYOm5C9J72fBamS~HmVXCnsDw5V7CxeaSwWwBZ1nROXGU3e-9JX-1lB3FLlTKuAAAA"));
            peers1[1] = new KademliaPeer(new Destination("nn6eEP8mqldEwjr6c76fpuILWyQXNXWd0JIiA1qVoQPkNil4QSwZsNy7sBU9sIBL~7Vxz~QV-d6bqzVkmVfQN69DUs10Gcd~d6ZxzhfHZb51JVM9nUhNO1OMaRfc3YMXWrnlPShYlMXtDojHtgJD508mDpKQnRlws1fO8ngIzfdnyEv~LCuqbxO9~qCRtD4KwHpOeMn1Rak6dEBb2RNilzQsdBcEpL8hfyzYTv4EHUXs-TUSnJTto880N7CH2RNw~dWGUFG~MGwEquyIBDQ-5mfdzW03eIFXBLMdleHaOi49zggS-LlGpHPFlLEQyitz8-APmDuhzpfu0Yv8cZaBby~tAkIpNNKqmwdOO9A0~~E~rHWEIi5KcuRd8SiIy99kGqeCnaR52WfCn3SL-JkOZFae2H-OTh2gBAgNp~WwZETY45qHxnhKOXfZ90vcfrIXBWqYycfXalUVBh4-nKrL~688Yogk2ubbc5yKluDhr0ZN86xveqjq4ynwKIcWTun0AAAA"));
            peers1[2] = new KademliaPeer(new Destination("kj8Nfly9kmCMBpScKNx5ZEtUvfKBpxqPVewZt6uni20nvdnSo-ePlGtDSOXIluUFLr-fNOel9rHKa1bkz8Ej3O2PcYzDdT7Twg2x6zlRpr8qm-k9mC0t-ULyT6edKlBjsxsSUQS-ET7Xx23aU~RP7IUVR9wf0EPsU07jcvDgf5YQXmvC5bk541qsj3JYJW-szAZs7B4ITDha33VcNc0TLY6--HgQ7HUDopQBbwYpXnMyA6x~0ioPcOoZ6IbNLBUFKuVSAo8Lht2bEPn1dWl1~k4v7r7ngmQpckvfBOYWVXKXQSfI45xBqZAUw4~lFJ3P7hrEdcV5unQ5A98e76QKal6kcoFp6uWAwTzeYiXaEqdGANd2Eb1TW-g3mpTPZin0l9OoRJOmWNpl2y5thhqmBwVu8jGT-ykPNMa6REwisLznIxyQWc96OAnMbQwJeGbziAe~HTtkmIPRRGH2FvCHQJKQMd8kQ9E3OsygTypQqfu10tBwDEejInl9V3D0BohNAAAA"));
            peers1[3] = new KademliaPeer(new Destination("-WR~vXesvCS8ADHbl2v92lBp5oqMHf6h-evhD2CcdU646v8FQ~DK9usNzIovvP1xVbqSoTYmdMc76HtAYI~kij3zBFgU8izfhHFwKRS-~lvUaZUqdIbMv1rmV05vi1zugyQTGB7ZAXhpx5s5F2Iqrbx9H8AiT6kyZNNVpBKEZjly65LvODQnJEFe4N-ZfUAazEFRVb4GhXdka5rWaVrBHrdW3F4OMKTyNtTM13YMeW~Bk3D1fPfu883U2~Vn0v2kTpAktcm7dC8mlFuc0rSjMRzhfACX-CvnWEXmcg3iBbMLhg~Ded3FlER4ByUxSX5K0fQ3PJfTj0~QxomHU5zC9hcqVaIGsmnyAf9SClflNOQ7svMoBCjsXHAyqMqNSrzKUIrVfaC-BxJMJaonaBAvvfxEMCG4PUO-G4p5fxQWbapFyYvbugzLEwChnIByw1yYR6pqpQ9KSJJXfAi59Y8GUWsUK90eD~1hPcNHK45iTJX8adTEzGFfPkSv9gnJzdZ4AAAA"));
            peers1[4] = new KademliaPeer(new Destination("B96irbkDbwI4p~0GaFTR9pPSE3xXUQvEFC1G1tHqvQcW0PyUVOeSMW-58aHN7I-dR3HVvGag11SvvSH3jhIncxKrnbM7xHHhzmjlBknBGWdZurNWbVQAhqoUlupKez3jDurX49qTKXvFLViGG3IEhJ5BsmQ7Iq2Yyr9LNXbPsTYSzo0iECPjn1Hirp0bp1EgHPMktoDNyZo~agPzGzoir-shGOYRjjER9dsxSyrhR3alXJuhs00LRlsUcs0Qj--IMbeESya2ikYN8FGAd3ZkfmEkjQmTvCCic46yOUUrD3EVBuWizzQ9Luhdxy8SxvKj6jwJW7KV3nPsCIqZNphgAZWyMJAQRkXfHgF3xXRfuVE3EhCLYu5xgqy8UkFUESkDWfJ9tUNmOtUHoPwCko~ui1N0tUf0PqQHIlXLGs7pbbm500NztsDktDlQ6lx4QHmj1qpgYHRD~a2NQrUycJatZCsWC6x6do03df7Rf-PDCEMqMlh777H~GrRJuR~wmjcPAAAA"));
            peers1[5] = new KademliaPeer(new Destination("AH1rUf09G4DJpAjwurg24xQi7l5NZzy3JeJ30Wqi-nKpR67pqjt02GbwDsrf6tTsd380jZXnkeNDwZESeaerxb7YEPTMHt3SyzknF75ytz-peb4byDpMooguxA-b83HFIfXdxu~Le7hIw~C5C3HZAVJf9njgYUIFDaPjlFOye7w0v3OBA9LrToq7VS0AKwMaMviwRNuZTQp-r3dPCTSQ7UlnorYF2Fulv1g9ECFXY9~t08MdwQrd8Y7qYsDzSFSQpMiQ-U4vA1h-DvOtho71s2-LFEx4Skfrtx2n~~bFsOFuA~xBPVxU5VqtXKngAduV2SVNElmczqYJozVHH48PDG34c97amNtBcTrngMYLINTDT8fmUsEcpXlYzi7uRaTn1F12HnXD3b6gcB2unOPi08gXsbYs7Reh6VeFEyD1wOR7zQU303xKoc5vWtKbulIdRVwFghrOJ4OK0iPZUF9-mNP50zHWL7F4wx6CZj2ETH1PeOSOrd8JyFqqPphi4LEQAAAA"));
            peers1[6] = new KademliaPeer(new Destination("mGHDLBfmn7Qb65wPfmLEOn7WtvwIUoTbeRtlsyl6qBn5gDewh0RX3RS7j2yrrxaokTkXc0XXCMjPpHpO4OIrmYWO05InQna1EoOa5O4VxB3yTxLw34cuFv46U9LU7J4MMB5H7uJA204r5PxZjyC0Ht8EGQ4197HItX26HRSaU90KzNFisRT3iTEKzJwHvQ3YPSD8A9Bx507IcGWaGjF9~jutTMtbADntii1rJD-ijGB6kMIcXIbr1mJ6BKiZLm~7UTCgZI2yu0j2btLVA-l7aiHPVADNCYwdMg~UgJ58EjQKcy8Ld3iCYYOiHbUghs-drY1QsGyADZrgsYI94E3GgV6uwxE0TwNI7Ww86Yrcdh99mNKSB4mM81Mb8v-rwx8YdRrh60S5mr9z9MJMSHJdM7pNFClliYxAsfUCjop6tmE9rjl8suERFa0KcnW9xEpDgQdl7sz4psDD03xHfGaEa-BlId6O3gN~eJX2CGKTu8vg~dF7dwaIN0zOS1A0L7thAAAA"));
            peers1[7] = new KademliaPeer(new Destination("qLoEsY6LIxStqZEuxpShCFppNgbnrO00VLKuawq9pFQYrb22swcGnCk2a-g9yvJgQbDH3KEsutcCczJnzjXxMlY68hMnBW15ta-rzyE9E4tBY9TN4VtgOONIlCF-~mYiicKLkS5ijUAcLkXMng4seSSy1R-tAj4XRgIBTzzmZFyuo7NamCm98DDf0~C7ArgyIP0i6UtB6zQOznkIjXsjlL14qX8HtrC4H7Bgqn4HtKTkhptK7uiA7DvowqwVGmdJ2uFOC0~8AIEA4QX7WjWOFMiU82gKS0iVOJgjAq~siPedt3L~~ekVV74pddAa1f-DdNwsl-tSM58H0CGTzKqPtQgkzCX2i7DB8yUWeHBkwlpGkBkHWqfwVkrWlIUGDsvzxp9pVk7R4iMEt6O5ZByMtqeum8UOwxYRQ97DQA3I0IE5wAFvAS2wjcYfr0r2a1SGqvrWbp7k1gogs0QAmANmLxeiozWtTdtKg81LSJDbu41yWLTQUpW6gVom9E1gfmTpAAAA"));
            peers1[8] = new KademliaPeer(new Destination("33x1UIIyubQpoxf7w8noZEpzIkYz5geOigNV5h~kc-sBDIZUH3SqR3jeapVTwgcWzKhcncy-6LUjVKxyOJdpskQ45djL2glz0W4i7nkI8VBD-1zu0vTKP97Jtxgl2~GQ9HX1Awsp7GzuvZn8cN6TQ-lhxZ4o8-sTfFZ~WzRqRT-1fSEElpJEmxGLed4~lgdkSoMy6~W7Zqdz3p7SNo3Vvq9WzMtf9hyUtF7tDg4XbcDQ35UX626~~zofInrpXeQ5tN8eHh9v5cenwNgCr9HE~W~U~HJY6c~IqeEi2casXpvHUfqzN2IsUUk9fiO-YvQ2QxOXltJzZckMYk4duSadtnX9GKgyXFwXwXYrbjyk84OceCf99bVK8bN0LqovvlmjgMmw-qAEVU91HppXz6ejOISCrhWKUbVDJJYX1OVlQhO2b8q-WnfAuxDM-cFJxUmbFz3LuglJGxqhWVKZdqlCfnzg12PQOSPX91nlEOa1GkQ1nfNACB3bHm6T6BCeU5KhAAAA"));
            peers1[9] = new KademliaPeer(new Destination("1Fu-p4KKjgi6EF898VtZkKXP1dw5nhDHqLO6TJVPHNTBzh-KK~n06h0Z92vNbsOYXrz9Xb9tkk25PDUfeG~lKJU8HlaEhGfMHpMkvU2iYAYdf7NXY3mF5~xpMQbBuaLtgTMk8ORZNVm2qvAGATmRGWGp3KOgIBF1g~1c61~iAdLiXOmFfW~Otp8TREX4SIwAIFZKtxfOFlx~L3c5WJysLUq~1tfqgqvNA-cfmx4mSsBEoVzadBGdJwgu79xrOSUNxjkmjz7kD68ucFGguh2b3VC-5h2kmf9sWR919RugzY32163OqHaPkiBx1z9cClZK6Xy9Rb54vxGXuraY0Q-UZ3iTON8v~-UIMGTxu3j0fnpPCshRx8DUung2Vh0kI6gWPm5cEL7zqtf-sYUOdMvUg3E~gaZSxvU6ZlcONyc~0yZQoKauEv0mjKfGNbjgfy2er3rr9hW1RWAY0rKCSqZEyIqVhdusvU8EuvdiDR4xBEUgANrqygZsFKqllXZNXAbjAAAA"));
            peers1[10] = new KademliaPeer(new Destination("a4leKD93YJmKQ6bXk62YOR2R2PM4upJ5dKljnFYnYSBbd20eFak09yXUJI0kTst6iodn6UTxUquV~adRXIsm7CXHY2yFIL6g-wy3oQ7no88zyHYI8N1oz2lS7cpxyJp8nvScG0PSV1t4GoDt8GISaPnbrgF2MyIGTE8RaiBTD9RPbfnBgRfZMbCYXkroKGWbZWXo~BecZ44bHSsX4URlPt-tmQMdv~vJHQheIbYZOn1RTeTi8YViMuSrEWnup2OFeYQ3~RPChCKCxybgD8XDJCB8RRwKW~bfPr1gNtfHPFxeA-Nh1Or8AVAU34XYN1aVZbOkeVXHYc5Ia6j5D~pJlowVW-vkaEgjnl1uCS7Cf0vl8CqdkaziV6IT5mYy7kRFl0DQSxZe5t7qJSvzHEjppoeeZQuhlZOYIlAjCUzbrd~3IYbFlrg~M~MYLW5IH348-djZ38M-8~iFI1yel4dpA3MsLZ82o3q0JuPJFClWOQFKV-HVhUT847qMIlNeCSReAAAA"));
            peers1[11] = new KademliaPeer(new Destination("a7wn6AscZEezQhChrlzYSy~VEKgtSoFGT3URO9eeJgarM0Sjbb3QVvQhzR1S7RBRYz5lh~MJe~h95bkA8LQ70Cf~cpguDXCGOdApA5pbBJl1JaJMtCS6x7fOxRDkONXlN5QoyDST95pwOOUjgtENjfwbpsSk4L1ACkhA9yGZ2aaBstL8ptYsLKC4O~3uuwiEiW~V9s4C1tFbmKDbqkH6sHOTfUNQXM6cP~KHIXXZQskaFx-wtv8bC0FYqHzVkCTi9NaPRNQocYMCmF0tXtPpyfH3n3aXxnJCK9TQRq1EAFkFrPDMtcpxjeDWuf7YaBX~srDxhc9tpvM4au3LNrDpcEgbpRRplOvyOIKENdxxE4e2XmDjD2g0-RJtPe0bUztcthGU5NH~Olu8EFkmqbCTjNDYGDdCrXa~W-nh08J3b2weddUUg4AFWRauSqvZjdBDZuLbayOsLca7Am6BY5ju-YW2IgPFaU7ZZZuQWoD~auxXMQIXJ74SXsl3CTkeXPTEAAAA"));
            peers1[12] = new KademliaPeer(new Destination("QOGdKBN7YCJSiHUlCK1AWNbC4krjrNgEbeLcsF7dE5Hg7JtHTF9tH4dr2ZJMApa~AVGs2lRljgaVIbeSG5j1oJNJunExGyl~GjauLYMokRr40GehupS6k2XZClpj0OUWK4~FsPm9hnb1Q3tGDQAC9tGjL44uPw9K8TM0YKhkWDlguMiIY-uJBg6hC~RhkKneizFUv8JJMk3PVL95voKPP4p0bEIFrSMjZ8WSE86t2bWp73g-fkB4ZTpDq5GGYgNeIqvNR89qCWBB0e7ItaVQCt~3xM2NsTBJoMVwkewdAJmqR8iGTBSqCB5cDt46jRhr1JWyaUDqDOXhp5CjHCH67lSU~z1p412NQg2YVDJZfbuWrhO8AHurjYiQFu9GeHgO8T0G0auBQ58zgIjT7PQqvvol7TzGy8LzSNdI1eXoGnSq2xK6FUrEfqNQoRgMhN0PQjJ2BpkXq4ov-CYmdemVViIoHRar863EI2oBJu25b-asSLB1E0puzY6escUC3KKVAAAA"));
            peers1[13] = new KademliaPeer(new Destination("MGFxaRkreqWH5PWcKjV3qQvMa3PE0Ji4ibLqYYxep0-YWNLWumlTHUAyNIKxBbEdVGKehvNSUpWZmAhSfuVyp4PYUaCpyWNQIdPOwDcu5DcMIX5DKC40iqgEPKtrIbWy-ZCATLC6ZaTyRDkIfVI5vx59y4OL9W37YF5QSQqQHCZu-XWDxE9mcC~-WPKR-VKX2tuQuX64UlFViCdwBZUIe4NRdQ~Qq1NHXNsYq2GfKEWzbMiGsgy543NUQqCR7wQ86B4~QWrqgwA-ageDQ-mMj89S6pFtqzQvFDJVeeLaPgrJH59neQ~5DV1no40Aa1DoSDljfY~qvwowDThHS~lTJETqDXhnWUkWAFngzLvfkIr8Iwk1TFNAjvJILND1XgDXuyN7cxN~RGT866iVOMbynNXT~Qjp95C9GxE0QjVpf6HMFgA20J-q97tn2eHQblK8lqnfrR3vXhWYY084s36OuuVml75N2Qj3caWcv4q9Cv2UGVSwo1AP1kCvOm-6Nez8AAAA"));
            peers1[14] = new KademliaPeer(new Destination("SpCtXvbP1nGfjAP0QBHj2XdC4wNBg2eMdHvETMXQ4GfTNa6bqYVQRGU769UHj4NdSjtdnQh8ZcDJD3jWCY2mEF5UXudzgcjmHIVv0wQOsMk913Jcv9-1hdqNHjjRnk8eQ9hCEsaR73XsPxo6VX07pKyvdaM1JAZwOTbpF~rIzmZFbGG0nh~m9CgkHdI2M0dqBB7qnRFJo2ziLzB22qEHQGdSCoRRZOzUXXyJC9rP8ly3wa8kH1n4wz1J3LrvwCteSN39KKvY3MqgzzVIrodyTCwgi1PVEy9tHuW8O1ed8MtP~~kbyhAAynw1uJyhYqbF8t5vHsWkn1a7IZ4OKoVHtZMEfjoEmDaqvaw2iUUT-NapQPQrHXF3qTmSeQHXxCACXUu7zcBeBp5xaPzundV2n2TEn7VaBl4-7b4JawBjHsBlK1lbmsrD-0jEego2k2G1D5J8Z8vL8V-nvkxrFI1uDibADg4mz18FAJka6QtrwChPPz~dCTGygyq~oQvVvkxNAAAA"));
            peers1[15] = new KademliaPeer(new Destination("Y1lobj88qfraP2CK5ZT6SKz0p~cBYFdqVGdj55SfQpsnMt5NNwq4HCsso8Zv-pKImmJ6uHK3DicoQx6c77m9a6bbo41crXLTG~-HTeihJ1ApPx1~cPjX40msKx9GoKmu6f-butP6iRT12YInftZ~suTSch~P-JiVDStZ9QTIRpBYBB7XrZ2cEqWGWhT0Lu~KH0H67W9-dHb2rqChrDzFHYx46O5pW86UC4r2ESoqEurlzknYxFkOjTEyaIDs10jVr2illUbrawG7K~H7nWe3TxwmsnBskKluc7EVXt9JRHnewOt7h53aheLPgTJga~Q5Nw~AO1A0VXoqtbZwDjdryEtA3RUZP4WeZwYcNrMaOb3GaCmAN69kQLQ7yr3apO47YAm4xaM7hjpq-MkW-cpzrvm~dooUgPzwY5wVZT3DzR1FM-rZ0x8MumK5Q1j4ueCWvy-VTmmFQlk0A722kDKwOHIvCN1Q6a0HSkqBX1aU9926U-Zy5R5WRghU0qatztChAAAA"));
            peers1[16] = new KademliaPeer(new Destination("KCnuWHXiTE9FWwF5N1q9YQ2Sc4rkFOMKoaxa3GLQzaRae2eNBdfOBR6953Ov1B9N~NbymYuAFNIa8vB2ipp8~Zvk6mCtGZX~-DmsuZWrCSozS9B3wkkyRN68fTtVFc~~LVw3cOOeu6Jn9tRXs6AjyjIexKMfOVZEd8Xfa4LRactiQI5ttnDuJikFBmBOOizxQW7kJPcYeIx-0f8xgMP4tWY~6zAR81rr46RAg8GpoIqmoFvMnxzgaO0Y8eNG1o7t1~obYgiTOWU24~p1UgJw~EEcPaITkWixO0OBq8luiGrDeW4euEvVealUjpdkAvTRwYHNdhAjyQWyvhRap4CorWb-Qqnz-Tpd2dVa88DUYcNRozl787pO~7c5Cnw8EmbiZZlZDoDbgM-bElHHcKXt~pi-RPN-1lYaenLl0hQ8CGYs5qYiZqWfcr0SbVYypBAl5ytnWAL92AAxlUwgSJRjcQZJHpNFvXdB~UNaZ43Txe20q7Uae4USZlixBacXXOI2AAAA"));
            peers1[17] = new KademliaPeer(new Destination("pmv7g~Q6QxvSKAnjVz19pb4powAD97mMEsYyq6lss29jxeq~1W2paXMo7rg8~2PZmrevdbohn1TH3QGXcSQJmR8P3ljXSHKv6LPRVPkShW3UZqh8BnAZVyBJTN3~vCUpcEl2X24pG0rPfp4IhSI9Jec2jAhvqgNhryI84Ja8K0v007C4ZLq~0EyMzGZOB74wG7zDs-xkKDdm1N5KSNIY69FN4HWXJpprrnrRjO~jP6Bp9pwR6gS3jomuMHEkAv1IjAS8QHytMADYa804g4ikieoBOK0KdyewWxMuQ7vjGTaFtSLjYDwF61OvGrkJMyAjD86sKLOGOtVSGSUvNIGTD12aNNnFWI0NZM~vBQ2YoXBDIIK0n3AdhlJ3IHpjWhjNdRCMcAwZSy0Dljrn3UUx~Z243SVwh0eKZnmQoMZhaHOkcrSNbHWZSqoQgX81kvr26qpm25JWlA1HAED0SNydXDdJ3E6j-PqVq9VLFqLOwHxScBNi5BOeFehwwjGhMa4qAAAA"));
            peers1[18] = new KademliaPeer(new Destination("GbunpZ12gl0x-lMhCIxxXEIgEYhbU7-AXryXI0HExoDHd9o-fnrH7sinjCAEOuPj2JXWU6w2TbCNcEUTIU5ByNMhd~jkr0nC6gXcOkB5b5VJYTK8xMAvGW3fXDu0bSLRyJF~1nyEhfsTcD2ktXcnnbmHfF39056zbdLWWZVqdzZlEbvirafum~a3aGPnNEq17Xoq2GX4~eEghb~bQR~lI~XfrC32Ti5hcxCvJzpx03JDB3i0x4g6R7WCtVZMgTWgKBskBwPwqDyoVN8YhvI~3Ff5OyZgvjRMaLFN~PBmC10K-jiF9KeAOJ3UU3dxRqYm1vz61P2PoQEBo-Nzvb~yOif2NbdApAPGS5Q1EMJ0TUHOglh4nGXrDo5RW5IUxMq7xckcmTOH3P02caLysREzoGCURmcHPjasJbkWXTjWksIyxMTkejytR3toNRee6S0ztfAmj8bBuZlpXo37UHq3ZjOxa2nMVsVKsiYFZcaLUxwwDfDZyMiszzCg2CIR1l3pAAAA"));
            peers1[19] = new KademliaPeer(new Destination("M4INJgAX4z8UDixLgE3-nbdGPHjlEiFxY02whbZUtI4I0y1RZjpfWBSi5EImT9IaX6Yrr~HBGtYjHgwdBHMIWAcL1RCP9erC8ETLIS6BsZ82ZQGDrTilytADIlOHNJH-6u~~d6~gZGam~-JhVozD9K-MQ2fxjc1Ev51PpD7zXYcJny2siVvD8Z5vvwyR6E3XlZV-wwjTsgKI7LlxqT~I5s4lt2G0m~tkmrpXqzlNQK3kLXj5~tUP4Cvl3d12OB5sdmex81TYC9HH8GdS9NX6ZMF~qKonHIuCNl3rG0zYFhp8oakS7tGFIIcpuoMlx4RWDNPM-GwNxPhBt0AJY2ZFSFlvVvqQRgZgnUsJbY4sIeXDf0HlziUgwItbag3zbITmiuFGyKrkXZgbHULXWRbcfW2ZBOGUF6WcKFzEbeW3y60H8lPrYtZv1C1o9~v3srs1sTmZrZz4d9y3W~4XozbWiIrwPpAbyto9U3g5QhTzQRK7vbknCU8Lee1QEStu1OOlAAAA"));
            peers1[20] = new KademliaPeer(new Destination("hwtMFQ~WC7-bZ3vINz90q6qTQJHqsbA~c13TAAbxfKTGAnR9xUOZbk5bX9D6wd6owKtnVDUfjizOjES8Pc~E78wmyckoMwpEQe8adZzd4obJwJ7HN~cZ2artJC90RAcRWhWRSpDYGF~U3EB3By~wRzsHAQj-8~gySnUXvjonXyYn6chcnsvSKcQvjGnovblLK7ksy3TUlN8~8jGtgO2ZXeQjtRWSH33ZBSCXO1qEXVAFoRWYFe5lq8QDirjXPdjuAUTjD6tmWsNXnju4JWqpIPcT4Yq3mrY56FASjOHaeREbv0hIRPvuBO2IcIYl~isG5LtRkfvph6oR~iWvTx3YVDQVItPOryOOafF-x~sshUdVMVc8pb5vW2ajrNLiUozkfofYgApP9gtV52dUySb0YyaNesJ7huF3d1baTUOa6LVUze9zbXxnR4lRv11IP~QtekdQDqxAg~c2oyclIJhR7qxZreeIw7z4yMxRsQmWS94TXCNUf3JgMS1PmyPTJQ-sAAAA"));
            startId = new BigInteger("20165322902067810213412302882682490742329252029515918366652609455164745599540");
            endId = new BigInteger("33702408623411834034558105528091829985375371188155381652341683994745035632399");
            bucket = new KBucket(startId, endId, depth);
            buckets.add(bucket);
            destinationMap.put(bucket, peers1);
            
            KademliaPeer[] peers2 = new KademliaPeer[21];
            peers2[0] = new KademliaPeer(new Destination("S6D4Rq-nd4bviA~rUIw8C5mqgXrmcxNHHYRK2jQGTQxYcOWisFY3xHCn8eSiESUd4yQoeFdcWdzoEETZ5EvE4Em3x9cewwJowM~yB8W4ISh0Tu0~uUUGFNQF5uidpQiz3KZ7svda1eVyIo3nSFW~-PfsJQPH7igYHZ1Fnv92vpfzQE-RGlcx6HWlC6B6FS~zhQ1DfKHVlT8tsBh8TVQ6h1uN8jv3hcE0ZNQBuH5eXwCitA9lUrxjj8Tjd2u42jchmVbGG~pH-QqN9SPIaASIpA3XAP1eQPjtIwC30GXG3DoHios9-3x1WtbD-bTJ7Edh5p12x5XqDETXM-4OvzET2mqpDLu7U~YD4oIevgOV0pK5LUlIA-AQR5kMtgM0-T6el~h5LoEOQbKcY-XZcxf6t9aKsUogIdTP9dRPvvmjKfBU03SoFUIzvyHKnKjJL4Ltdbcy1QRal0x0xcsPlQEevxbtDo4ccZTBxoP27ri8lrVz5kyeTXAyTjYpwS6DwNdFAAAA"));
            peers2[1] = new KademliaPeer(new Destination("0c2AB3pMEwDR~-LkkTuhJtaVZuRKeHMwuqGZLnmdsrRUo9d4A6u3daVXV5Thg6rXtFrnLNL6hBp95eWGeLcjg3C74xUymgKnBa35n8M9Pb3jHWf-feiE4TL8H3W0DbYaX4EKwUHfiRB~RBJ2S62vuDRf9Iobu0SEio-6jag7kOiDIpU0O0JHq9SE07JRpC9ILlJCQSMpzq99O2DoskgxtyYG9XLlGkAK76XbPjC0BOI4ZSYYYb73VU95SImGOxWtIF6kLt0XNH9221Q-P~qF1bbRrdMtD4cCJ8~kJru62dV2RRXi4BzQF3a5GXwJzUxVEf99pS2dFAe1M0~rJjB37BUpeedab3b2dwJeBLKGu1WxYb91zPmJDF4gaEEVygJXDmeG1uo5S7aO7ExNA4lMqU7oDq~F0ae8jUZosDVv0h0EQj5vBledLPR12yiDpHfncEcbT-VFCJacGg-ctGGatMR60qvmQd5~a8diWRvIdy5LYK05afREOrUvm7TOtxJfAAAA"));
            peers2[2] = new KademliaPeer(new Destination("NjfPVnZmg1q42gomPhPMdW7VqvYzgOsvLijOPBj6ZBABcyvNi0bpEx3mKAe49-W-w1jim29Ptp3w3M0qjTmQXyX5gy-sgfhiT-Daaox5ZmxvNYM~3zfOXbBQOGLJYolilvQw0WNNb1TyCT-q-vkwGozdAQkBXKb~ZQREPx~xEaEc61pvs8MWzZNZndX01TU9QbzY284-Mzth0BwkHFm-r7gI9MjnbZzhWYI6hZzESmVcCzT250Zy2NQxL8dRr9DT8nHdhLic2syUhNBL3kTegsLZaNCFLv-uF0ZfWU~Cmfto9uuUhd9w3lU2BeQtVkYBLWH6UCSJBTlVhJF~KsuxxAJljKgI7QzHXSlrWeXTPI7JjuWtutZ-MGsVqrPN~NbvrMdsrwLCK8Rtot~6Qk9W8pxFmpYUZ0oLl9XuqqYoQaUxSbDqcvhc-28BWHRWqLSwcTGQOTozsjqS9kF6aap4kX7wnWYW2NpCQTlJNCgHmkIL4MGXTowRlmB78N8o2-JRAAAA"));
            peers2[3] = new KademliaPeer(new Destination("KAQN39u3gFBVqUAUw64tELcW7aPxv2MwpGpDauaDFjKKEnM-~mCiaaYXX0XczAwDBvvuxlq~T9IAXljTUqFu60TeX5tHaQFJ-vBRxCVEbbAi48l6H-ibOc60IO9pAmVMCp65YzmG-bVJs8P8z2qX-~7TvzlzZDpyIJCmkHJk-wQyDcsmx17h5Yd6vAeaRt3EjqWQRAmMn5Li5FQRJql7wQNF6TeE61Zw3faDdXp~-u6dM62~ee3iX0D8uVU1Xzf7U9bYxbgyCwiILVSdn7agyEUsBR9cIpP8fqJoj3WWtu6TdBYnBULAogijBO8n0IknE09j35tKQtLynur6lajK0gejzbzb1cW7SaVzmIrb7qUPFdLhQpZ-FCoTFWq0r7fr7Mmwxi3CYE77--DOEgef0tzP~nRhi6-iBzaHkwWkifPJs6xPICmp4re7Toedyd1tgfkn7cbKE4tIU6t9ocY9D7Kzlo~g0DKgHoKAxwLoPP3Gy~wJIJjL9VHxO86Nbo0RAAAA"));
            peers2[4] = new KademliaPeer(new Destination("AI0UbuCQbrVbUOV2Wq2kICJr1I0MgYzHNFMlLC2hr3Bjq81h07Qn~ulwwB8GfLt20E0kNJc9v0CDJZ~g0ZnIE53A6zalFHl0PMzsVByVYcpFxk3Nm-xR61Bnvjayz6u6Cueoq-tQe33sdgBzyJZ~gF7IeP-4PNpQ3EZy1nkcatX6h5ZzC322XtlL4LJAjgiWVmZZ4ruzEchUKE7A~f~HY8BMeJ-veb5j4xCZbdyQNbYPiMtPKHCkG5dyCPbSPSmzSmuKLPumkFnTgChXEE~KjF4E~2LsPIasgVMKv-0MBhpsHCVsxe0a6APFDXBRdLesPFVho8eQVZ9C5kkYCn9R8JZVGYtUZbqC9Xrg1uW4~00lP0fFBCrKFnxsXxBS7wrzkOM7ZLhrnqMJ681mTAtStZOeMB8ytKuc5P40Fb9~2HbWj9ctdw-YjO~otmARTRV2sQub-WEJSbNiaJKIQ2D28HY~zyX7-~oVvqPY0~cgb3zVw5ienEkEoQDjhaGEaOAaAAAA"));
            peers2[5] = new KademliaPeer(new Destination("yhJQpvlDPHAkzGMy2oKV9lS77hrHvHX42BO0tpR1GHvqvNy0u9tPMZe6EaSxE3YZecQg50imcOa3VHkH8wkfslnznZli~TR59PD2oXzIoOmIK2exbZc4xHMID2pG6yXvPIEcS46s~6FyjDU72OdSLrA7wW-v3ABxRIGI-3~PEFirtwVAZPjHJjiFH~LWHL-CfV08xRxZ-ho5zrK5pZcKe0FjFyXSnoI32vNpztxTRoJ26E8v8Panu5~NElkS9SIbMCdrl2fU2ZM~-6EIBB0rrOmtJKvUyHyge4kCPnQhzNUb9qOsxS9OycqGYx4XUuzQV32NsgVVbNorsBTkEmaIrmg65FuTKs3U-OEURrUftf5JPrsWogSkRWRXGVielYg2NAAYkprqWZarLD4VF84Ro8q7KBx-l2FTCJtBT0UrYLgqMhnhka5zjPq5rk0Ja5u78tE-JN4snOeFL-VasbsnsJBJaONoagZ~tsMlRk468qI~jdC8JpbOcXOtMHiep0OHAAAA"));
            peers2[6] = new KademliaPeer(new Destination("uA415zp-9YVYHCdbRrFhmzL94QxmO~J0kdV5KqDCvdbMJhfOsZ54Ch6o5SMBIkFbPpu5HQmTr2syBqc8VIXa1dYNMvkyxmQ3E3VnyT8u9mxOInantU8N0uDn39abKHYPSgbc1H9Wa39doCK5H0tlFmPVTVMcpEVADTS3sgCClCxhNRk9jysPfI6Ng661w0I6ToKtU0oS~EzqUXYwAzmerTxZ10v73wwkSkdZuW2Szm40Vyiz6jskAjYh0HGyjcw9VWwEvBEJm8~cm7MpAvyjA7kmrtw9lXMwODdOiQKsbgi7P3o~a3XKl3F1EIhIG0SRiin0sDBbb13YgyKOoem~hXLDHBapYlBGqEurhRx9Cur3xkaHMFT8DGqwppckOH6kVwB00MQN6w5rdEKFgVvgOnn5TAl5bslIgShdv9XEY2kVd9Iu~2B2IqyXnIIq~G~FsrxurkpqrsXBme-~ucH0PeYPkwJTbXeLTbiZ0LbRM5GEqcbDHCK2AMAClQz6tZT9AAAA"));
            peers2[7] = new KademliaPeer(new Destination("lk6tmG21Ukxtrvl3AJuNCtgIpYV45WFsuqN0ibvClU6BkEEBB-LE6rlloYJxEM19Vds9YMQdXzNfh2l5le0f0VpOJo5b2zUA0V-nSNbg1Yw6aHpS6Q6HFJ0fPNDrr0fJ3g3J29VT1oSscFork0YtIYVvWkJaStypUL0KrgSLvR7k3dj9GufRYpZmVyA4qUbNFB~QH5FQX2uW6wDH1hp8NRhOPOUrYuksNEIXBh4uK8GDsOTIY2RRXy8TWaIGQHPVFnQBvyrWvZkPUiQ0wWdprdgJEYMWemFRAevbJQWiRaROASm7EbwUMYtbSuCPThPiyHiTy1qfh3IaiSCmvZfb3nzNTBj-B0sni6IHsCthZ5Kp0Sv-K6oDnynR22IneOyOjYauP1MJNF4KvceqguFEC5lMiyfAsNKsmkVIfd69ZdzEZhPvseZM57yAkIYS~pTf29Zrp8XsvUWl-jrxMREHmFOluQbnibFXKWbf4XN6hxUI8ATWvVUtH2sWkGwQv5RYAAAA"));
            peers2[8] = new KademliaPeer(new Destination("81aNQKdLBykDOiozwaw9dB~-jBuHTzy-hI9PJTazDEBaGQ8Dd0SqSvc-t1eV9SwZdtcNfvuKsPmcZCcTc3KcCN7elQ~9nFg0yjj0Q4GRMrVpmpQbtdzAp8dw0ALu49lrx4cP0VZOJe3zRS6v1DluAXHyiaIcRVWNwUcBIAFqMHf17q~xGUMkNi11UgZbfDj2JMyrmgPp6pE5OyxcDDwysF9Eye5vEKsLiKxcEjkt3xHP-7YHEtBMN3LDRVWWryr2PI9I~-NdbpiVpEUJPorfU5Od0qotd0JwdKoUvSrpnXBBxZrx9DOyAbZAjpEv8vinjmFgtnN0fu6Q9-Hs~BK6kkBrkhOuozm181~2I2yPN7TfRuODA~fZ0kuZwJ4i4x6smfIeKmhPvORMXoOh2QFMwY0N3nHK99tOJD4kUGRj2me19lhBqcygUyH5zQ74a-H48mwEhL~T1QS0anOwyr4qSU6dCBM70TUV69ivZD6l2cdzWHGtLZ7yj7SSix8n6mRkAAAA"));
            peers2[9] = new KademliaPeer(new Destination("xmViEzuvHm8jRHDTfsXySzCvjZ5NrRC-UNjIuQKRtqwJml9vr6QOjctqDrmW6mBf-VfC~8JULqTSPRZANOPgK98OnBAw92UVMgDP4I6eeYENP-X03dG-QeRhdYMpHra9ZieO3D~toiKtv~g2T8JzrF9HMrtdzu04Mjg--RcMEkr2Z6qoeJTU0w7HFvNBOKf22zbM8-k-75AJ94dlZjSPFJI8sQ78laJHs9~YroYB~9jtLEa8QPlPC-aDXPcMN1nhokdOrpBVmn2saa2nw~05m0qmNNzUbeA8bBhm2Xp6xkkA3O~2Th2GR~gPxZzu7DbLDJcUpv~5yTkqZn1MAM9wSxDwhpcXDcP5kQXBjKpNVdaLksZry10oq1DJDTyPa6ppVSW5nBvEAEu0xRNho03D78a2tOq2dYLe-1dTTRDvfMAFdvkKdjGnZ2PKIHPx52oshR3L523O5GxLSu7TdMMVXsTu~~R3yy3LtRyIz-X75P5FNwFcKXl97Kr~9dJK6K0gAAAA"));
            peers2[10] = new KademliaPeer(new Destination("JubQ5NJYzrc~~E~nUonhrq0tdU28kqJtutN~ION~LFuIeweLfOUpZoVC9sqRZpHlaHunRe2NDEAz~pf6Pnj-FY-xB72QCB9mjzIwRy7YRQxq6Z8SZ37xYe01Moo3urZenoIDHhR7BDXK-VK5D6fhI9f~5DADJiMUj7HrTjwLW6qFJVykmJu1PfeplUNhViaphOZidDG3ztVU3HDq8W73zI0C98vEOHK7YED~L8lKWIehINUqx4yt9~VAunGRX0o0bmkuhHCm1TrTw5~HpnPYehWbJeWEV55N9fRcsQgQbhd5178DFLDzuZCShFyY-wogAoXx~z31LRiyb6WWaBYIJS574gCrvQu9Nf1Ig2g2WONbl0Oq0KU0UP9c4Ghhh8VLt9OWc4I5t2zJanQKAF~foOktMCmtt4Qh10ldbTSLsE1xjeL41A7-HeZGa6vjY6mIHCwVBKFWKbMGG4x84bU85XojIOzS~pyvRvfk36bwBZAYeqXqVnP~XaIgzcHYc8zEAAAA"));
            peers2[11] = new KademliaPeer(new Destination("0PFAVND61bJG8fS4e5OqVOM31vvtv1zgapqoc6TTBYg17w9lmKJkkIMU5vjyBCmyqYlD6FMoXP821ah~cUZ1xB1WGqTkMiYSEHZPq7QFXjy6ZZvoO0aYXT~aytqyVVeniBUN0mhz~~QL91Ak2HQszdwQ67YKs-cKA6I9CpFQUK2KNSwYp6DcawDruvZCRQCErZA-hG8~27pBB2Pib3OQ7yk1-P~lxGixJsyBzDkIMY68BWHQA-gKqNa7Se6Ia0y3M0njTePZ05vS6afxkPkNzi6c7IidPcbqfZJuTWQ72CpOMtSwjZEgWKtpwPreDEwz5B57OjfGmYn3HiBtrMlzFAXR1mQLtgMd7paOCkjDsq7f-Qu7WlIJ4XwzYNjPFt6Yiw9M0p8cCTehDUb5It~C9YYTBpql~QyRcg1xZrh~33b8HgMqLwP-NhmBy7vLu7bb-b18S~NaMdNOkD8GVX6ry2ktPHE3XufIKr4oOfWN4yRV-ZueOoQpimszSLAjIWc~AAAA"));
            peers2[12] = new KademliaPeer(new Destination("ktoj6E07Fz5OTbkqRP1IIRGh18cobqpdHyEBzvPAXjOuMAJL~NTEiCuit5ZLKRspLj0PzoPzQNZiUFlUfDMohLilIfWOUQ1lotzEHD8yx1uQ6VAeGG9IEwFuAFexIwG9dw~~vdjajzTp-J7430-J26GUSOY~tRFL025WPOXzL1S5iV77tapRWYfIBFewZSyoqvym~3f93emUfJnHZzLCSxahNSqKlWBglCRUdOm1c6x2QLpi8yVa-O5X7qOFqQeKFnbZqIzsAtSHs78EKDNzbjH26XUX~j0shhpkwQGy6co6zvSUiIVqiceR3Gt9kSKip4OIRTmojhZoQCXw8jWUCIiYL3TNWzM5tTsvMwZa-sW8tMh05JbcW0zydT8GBCnZapzkT5m~IRWt2Ukvph3jdsIF7cd-yjo~OXI0yfr~vech-392aBmDEQ-P2UuhIe2msyuhl3-nL91lKxCDplZETETdg~QudW3Fsabv6W3skXmhkg3s9HaPSVWdlLP9bb8SAAAA"));
            peers2[13] = new KademliaPeer(new Destination("qeqdhGVeXcAd8OmNOCz~Lv6QAG4er6yDrIsU1q2VDKjp4GoLmJXaDaRKUqGTUG~42tmFEPh7xL~kR1BgkgvXcNFYtvFtstaZtoeTYPb-6eqgBya4Qcs-FqmXd-nqNrxnLT8~1eymxZLE~k5OGO-j~9nN9rIH3KQW75a0C35ijbdpVaIPtp-m7CBMhcsyaFJy1GhWL7mr0loEeM59Zx8pE0EwrZoqmWwbDYtqOA8VSm8eGoG23akfoacHuMzu-uoJE41UmN-4-bU15~sPLkoaGEszSsOmjlao4RObrMjP9eN5rhVaAXKINEp0dEqyHABvhjS6dSTQxCdEcNMo14vI1lgX7AAKkrMe2i~F7yhy9Anc5sAnMuzOGwpbFgXtrVssACmYCHljFznVmJ8gBlrdOwvAOJA~BoEgcSkHgbNi8MrS4jECtQ8JVfQ6JXG0yB43uj7QEJTSMyiE4RICtOevXlCfaliHBchNuaXOFMoqZDLi95vOWE0FbTNuvYBUs7sXAAAA"));
            peers2[14] = new KademliaPeer(new Destination("BotcUH3e-8J0TIjY6lTlBqYSKhDT6Wv9d3jCQkJWpLo7sB1q2zSIz9xFbH5FlqA6tsJ5~zkPdPn8EFuJA1UMM6UHUzoW~mFe2gg5WzBy6Wt6i90QS22dFK7Fc4ajlk7yH4L7xmlnToOvdaZ9KcfHpqXW6cd7YnyLRmfiyCZP4qvNIk10IlkhxjrUROSoCtOQGRI17tZ1PDp67w7vv0Tpg-FKpfRbBHOIBNLgwEpvEG4V9OZO4bxYzMAcT1Mb~drWcYyEHOvKMjApvEer4rIFZGXi1OCoJHGtOfrfqlp~J5NajTbzkk09XYiRzMZc3UI0JUxs~3JAnPyPWZftAtb-9IhaWt9Ai4z5~lNFdc2DkD~tu9p9iFquKSB~h86ZmsyiLoO9gZ0zHb6EXj93e1rOBYMG5qRk86WnDabGUwpma5V5O5US~ikOaALS8qHEvwFLQtAH2AbJKKN-COPpN9uAOZreTLM6nCvxLx9jXa9PDABIzTgr-BSmOyjsojv0-~JtAAAA"));
            peers2[15] = new KademliaPeer(new Destination("NcjlUlxn1nDmqPHrkmKrqEbQdg-RE24VneapuLS88dOgIqfyeTuboz5CrXrWH77PgwIMK3IA1GPgfCHVwkclAtBQ0rvHXb3L1xcjFMlkIE-5jB6hy7raDitOQWHsWu52VqmIwgfhWYaXJBpPnPNvlvPKMzDJFRx5ibAUPSJyPidXsTO2XDkDWwvruQtJt3CLBaPmssdzQ5mus1bB9sFeR6wjd0APDdizVQHdUwySgIKC-SFQGlFItKrDCjn0rBLdkUmd0cDlbx52PJRz~TPjqZ26FgpGk35h1HRgVOnL4yNyhd4cgwaaHy4xAWnEQtT5m5cIjJlE8So046w3pRmiB4z4t3nQa6RKdLmqsGpy-xzRv7Jj970mfG490F7Aih9xFZ2O6OZw1UAoEuxcQJ71MRczXLq1MIdJ-UQW1bAMHKSAXoO3VtDxsEs8fPd8nLBiiQWmpvE3bwC3tHVQElVvCv4jrJFD3VtRbtkrJKJrAQ~8Jb7yoWEdD1eQsCEDjyreAAAA"));
            peers2[16] = new KademliaPeer(new Destination("3rSfEOeQzMEfPl11FmjpN87oq8Dq2fdPfUx0CnK0m3sJzzPrYprRwJ7QyG7HqT-EjXSJt~FYsPzQfPaFUCeLb~NQHw9qtYj0~4p-z06PESuFs6cX5HIKqej0R0Pmjt0tZIdIP9x4vQJ5Y79c4W9yLd0ocglV2VSjTFc3Z8gtfdEfBKSWrQKiW4eykb5DHTeV5yl0-up71GGQK3Yn2O4KHFP5iObsauPk2mmq-SUvC1XEiZ8jCiLPeoywgfz0-UGn7jz5MZuTxlp8VFzBlU8Ox2EKg51UX1lkUci09ncctfB9Xrqx4glGc~JZCEzmokboIRNgHIog2je7q3V0~Y~YPULK~jRSw6LZqBQK9X-wZGlc~8ON7i-pyEgwvjepaxVHz3Pq0QI8C1iJ55NCWT6WwYxAqyEMdOEgeJVPThrw~ffdSwQvHJCyZAl8IhCreKmsvi4lTP2cjuYENPoUjm7naJskMrZZ2~R4mPRy3Pzy9GL20cIlzj78EuyDwGHQSmfxAAAA"));
            peers2[17] = new KademliaPeer(new Destination("lEdKxq0r3E~O2X1h~mjwIHAy939KiZmvqGXOMFRm8-2JdSghxEi8zY0ULtfyJeHSw4jATAq3O47bznnWpsHoQfKB7MWdlk~IlnSztA-DOD~fWF4-uPspOxE8lXm-bctqAbBpQC1zvLGCItsbL9IVP4PmYNsPOZ70quHPlY512RCvl2iHFs~fwMHmWTBNCMmh1z0qROjbMHNZLyzX89eYW4oRNjhISLyqJvCkyGpqGTn9~O6igdGMndr~e89xylqSfwBv2hpfphIdnYMw~eGdhOCu1YktPm6Ts7LG2L0pOTqId6gDtosU3selfvj9~nZDAGaAfqwgE2Tm2tYhHUo~yIEiWv9r6gqTQ15yDjUKcy4g4AvkwstKk8kiW3tcP8gjTNOJuEIe77ZZgOjniLfONIWBYGT55aZvfOz865nujhgoX2gjQva70R61WOgBmsezG0vt6HnWwmY~L65pUUpD-7pS~ijtfzSqo~ObRCesyNKApZaZEeDvymZ46Iud4WqBAAAA"));
            peers2[18] = new KademliaPeer(new Destination("79UHlH-CXow~cD3AyJBW4kB~n~jOMuzpeczu1HOuQ6d4uoCQkCwHnmUqPswnYUwHUDYT0Kb-cbJAwTit6nRliPuGa86nr3titLWkjj2wt6KIfA4RRy8-dCdWgeWs~Y97JtOb0mEtPTp3oxCfZvt71clkhTWwCUIGBnSWuq94V5Uo-m7AZ3kdFpuMBE-yvXV0hvI3qgXQB3jfGbJh3iUYBIHItpzZLEbrmgzixJvQaEKCkkH-BOYFOt1ov4GubnGPni-wO5ASv30eng4KbRDOsP5TT2eU3ZddyqFYYTNM~OYkkXco7xEZLgi3ufnSnHKs7iZaeJspYN6d5VNClKeD~w5h5Q1qEgj2IyOPqm-O7A3pbc-bOPIZq9~IX9EzPAtSAFvX9tEL1kmdX9rjvxUmUnAVzWGIvSv--6KlQ2D74AYNzT7hsWcIoxQl~1cdmHuxmaZZJ80~z2n3QUwsRYUnkEcmI0b9cc3TeloY89DWqDTE7J8gYhEL9GKpYKheTcu4AAAA"));
            peers2[19] = new KademliaPeer(new Destination("bQcXsert9WWkrIzjqdg4l2CvjiVRmDvEz7FNqfKqWL2gRBiFVJKQsD9HNwUDeO~pN3sB4U5QmsG~jK39qD1wRnp6iblFs-KR5KCMQ53i0fzJHP0UhUmMV1pHrElN0T2-PhgfKLWXlpO82q-oD9qcvHAlLXN716xTBdH9FslxTCpknm-ATuJjaWoVqvv22bhn8Pj1JbUu4f2yTlB5phA9B~5dhB4ssHqz60NSQBzeewWrQQnTw4JswKQQlmrL-N7aTo8u6IwX9VJjd7TGEvYT0qOvixBSLo2MaaJ6kZ9qIlOjN75ad4jJkWksEEQMam4288wRBdXE7npjDoPDT0Oq~BaSFzCTKs5QaB4UWldKIZKrP2gH7zwhOciHhwD2~j~LySTqQrGV9rfevnfYCZAyHDkDJsj6YiAkeNRObfzmmRfyayqGWPnfxs2DSF2fAmApZWlZ7cHh4tVIOx8QF77654arYlKasJz0fPmq3d9-ITkeFDChXFfNZ~jsRWNLcYreAAAA"));
            peers2[20] = new KademliaPeer(new Destination("wu3A7oY7WU2Re660BWbIw8eZDx5F155kwftEpb-KP4Dp6OnJxaFGUVRJBZwKPEEr5C62Tsb8cY30df8-yx-MbhKv0SVG5dG14Z~L2dviekRTzT1czQ1V-VzdnAZHbuy1uHHY2AiZVtM4NW~2yn8C3T~6T8Qj8HYKLH-RZ~vPGVnfy9CvpXG3YrANK9O2HcoQJZIqDR4jBsUY8aWf6MvY-tEao1IQWr2oWri0UMLxklHlznNoCZiayS9ZHTUNtqjY890vGYdVZIt9vHMRJpd7yLevWCekS3x6jEE6k4w5erOnYqo~jz9HERLSX0IPlpsRFEbXqAhdG0xt4HmDweTo4FT~sF6yIuPHGgKFBbtx0H6QK5pgNI3ldvDdvIH5w9UgU93zAlLIbOttXkBs-CdHXJpOiJoap5iZQD5w4u3X6G29mJXzY1s6oK2pxId-VdcFfyDOm7veCqnhyabH01LuxyKtEXEfj36xQV8MmCtVbKuxPbTgc-2KXOepiUY~MLQAAAAA"));
            startId = new BigInteger("54322108851495321962026285688661141713340520926301416800794348337691246625057");
            endId = new BigInteger("66306385230365087118809986298744845651941672849188218336247011280980080257051");
            bucket = new KBucket(startId, endId, depth);
            buckets.add(bucket);
            destinationMap.put(bucket, peers2);
            
            KademliaPeer[] peers3 = new KademliaPeer[21];
            peers3[0] = new KademliaPeer(new Destination("CXOZUPbspzyE16j9s-WFbc3E5gdmWh9wn8FQKrm1Mlu9QS-01vdLUHvyHt1gsw2dk2IhQeKON-2~r8kzXFH5ihGeMlj4Jp6NLeHPFgcdJihphoMuvzi6eQvH9SEnLuOiUpqF-fi71HowwNAUVFY82nu3eJs2b1ZIAhw0LbqgYuQFFE525Y8GmAIHu56xEUBSZHOh4sAYJirw5~AfeKf8Pya0lQ6KzOmCMZNmn0GrHFe9Cl2hQs9kxb7koAyE41FBk0aABJBYiBeinaikAt6H-vP8FqJas8MG5HgiI4NGXKBIfO~vJlFWV~nKSYSnWc5NTqNCG~VuCTyD1MdWfVmqOHSyAgo48sIxE~vmSjEqw9RG~usPpRDeg-lH7M~DALpWaAcoYXwTysQUy4COf3XVIOn-NHQteTrHGl4vW~A9~kZsTK6a0QCNcMUTFySZCMTtY56vQBQM9yMa~afWRthC2j9mkYh3Z3LqbvGDpg9v6PcioRdMiep2dlys0DsmPL63AAAA"));
            peers3[1] = new KademliaPeer(new Destination("EXPsYcvT~xasC4ODPnbVf0Rssly~YIKneNlwQED~VO~FJ9H1uAKeaWr6dmoCq9i0rb3KgGu5YbeEKOWW~dAYpamUWakGihCr9CZvet8Nvw~gVlxVSsypb5UxQjqePq1sbkVsmTbmjUnrV3YswqFI8gmGHKooR8TC2x~p7OjCBc-B09ZwMMuwfw-0GZpR9HcsVaWt54z-9XYHMES8EMfcIQSfhAxN78KXUWbRiaaNDTjiNi0SUBtdD4HmOLlDEYqIv9f63irZu2NHmjhAu3a9oUGzcv1Op7XvVm48IW8wzwoIRConfRwOyTaFF0Ea3N7shac-~hhkUWpdRz~kTPVy6XbIJMlFLjPCeaWRBBv18l57GV3eV8k78aqlMGDwioacXPPC8sUBK07wbVYjo3ogaeusRe92GqZ79PZ4tebv7I-jCBfY-OyIf6Fo-U3GESrJyRLSUB4GP0WVehkyQ9pXxJKDA~i7qPNTw~ELZfrUjvMkmI40XBOmFTOo1x6pib6DAAAA"));
            peers3[2] = new KademliaPeer(new Destination("wC6mfkDNTQ-Qwp2tSTUfnY9Mp7Ncbz5V-9DFSAg1aBxzA~uPlQBDP0SQKsoYFQdMK8dRHV~8XFIDYcj3jheGEl94ipLkaebfKOxM1J7jGQca7clql44CWovkALtdXsy3AGnx9~h0fIZjM-OE5pdhc9wsxSuQFrWGXlE00MOh6Fp-5RXYI7Ap1lUAm92zbmnsptgN0Ar9kDw7pBLoZpAwoCMIWfZcUHSoxnQi660tMN5LNhWN11DJxoydZjz16u~dt-jpc6~lRaEz~2iQzphAzb2oZgFk7qDyhKHuRQPOq5w0WXCGqlq12D-I6h3o0~TAs5tP2qWpbx9ZFYkvrOvntG5XkiphC7nWVqLFBwu9~HrrgOeNILypitlR04cFjKyK7dvu9PJA1UHsoSYR1bL0DfDkWHoGvoR~DUqvVv6xiqYgU6pXbCiW836NF1wbSM-IMWoiH8dwa9UO32gdffcZYzE90Tm5Yv45p-uN~R2rMjNtMEKpsZ4ANWL9ixvM6M8OAAAA"));
            peers3[3] = new KademliaPeer(new Destination("VTf3p~jqANRsnDlVHY7NGEOMiaHkLQrTxvi8crGLyrqWJA46GjyavQIgKVNX4s5pGJjgtcF5PAFpjTLN3IRjC3G6mJ6i0iTKhwE7D5UaeduwBSwsdSIjt5-6j8ltc~eJxNMa3kE5fPlw1sqfHwmghWh2653bRoxTGMd9Zn6i-~yCYx57VOJSOjiEUh7Abww15L6-OOkriLR1hLolTYATpTjNRaLpQbJ7vJRD9LRw4RwsZh3W3qPlY4ElpULQQoNRu-rd5GTZId7YMzNLZgTISIDUnOl0i6wwjxk7N2aTyy6z~~~t0dlENqPs0M~clXnMPLS5IwyuF1Vte1~cJ01EOIB~OxuQ--eEXX~fDW56BX7OIha105jmsaMra~IfDkeSyVNXWcmPnf9RH8p4qS~JNdpHi6W1RvPTHtWyJoVLaRy-Rpi-KDfcIFesCVqehkbC6rxSiAW14~ki4y-i~qBrMaKQS3-e2d0i5pt1~rY3gYrt~DIdu4NoqNty4y0ap81SAAAA"));
            peers3[4] = new KademliaPeer(new Destination("FXvQZlggOAdAaRpSwYA0t-r7nQdFtms9BcmWpKPiSiFIRlAklBZRL2Xfh97ZfLXE0df-pybIC4f3P0wRnJoCY6Auj5nzIScj6kVZ2EtaAaWjv7K4W0RTuZK1bOYXkMC-EdvkE9IvcvQaEkRT50JLFvFOCzyl~Pevlp4EPkDK9v5Dx~tZaTiF0KJn-MG7BAWcLQbaDhrpXz~B7~jsQtoA5Mh7eTM3Fgr85kjk~9Wh6FQ07aB~QvYQvGFDB-O1tgL2OaidKR8Xsc9jlrE0Y4o7S4Lap-dVtTjbi3R-O~EMJjjrx65SJnIpKx-YjIvDm0KiOQPRCNUfT8aXzELH1cXQDnyHCar3KoXzDfdNH3FKlh5w1Tue6kvfw4A6cYbr-U-P5aPp2nwr51yBVw9HC9HH~9ceUf61UVBT9pNt0sNUhtOJ4MdbBXc3UVU8ODA3tiE5cQHD~4BhUVb6yOwLrKn7YJ7tFWZHXTbidO4lkbJXNOuEruUtWdBmKXP9nWUMXstUAAAA"));
            peers3[5] = new KademliaPeer(new Destination("303TcM-QPg0jl9g-k~-10h5LGGxG9rbgywFg8B7y-W1OBtqQbfIHthsk0ZDzLIQ9AegvbEAcppLqjdsRoYZjpY3HxCGODblqN4L6pCu-M0ANxBCqp~vNzGGoGKsFkWTCE3tbyuaF3TYLTHWonDn9vuxap748TJLRzt3BzuaUVD9BmUdD66Zi3-Gtwi8IpGTBpreTBA5XSdoy1nfHhQ3vpqTvPQvYNv9FdehomunWNOutzw99M8eAmGYw1-7QVo~GJGyDMS7Y~lSMc-frwc5kwALGSzURQRyHk97tjsbBJpvx1Ftvp7xSjH6vYP9YHZ~6pYrx5s~TC~C8i3HNq6lJDHF9DEmVmLbjwyFJiq-R1JEIsZq-chlwBIxNdNS78nA0qM-uE~FMm2I3rhJNDx6VNbNBmzFJkZ38m2KMiy6WMG3EQYVEHhiTW0pLli1RaI05KxtMu0u8WZAqdJ99FDGeKRcE4lCKGh-0dWy8b~w7pj~dUmiWbEl30Dpio8WuXRNCAAAA"));
            peers3[6] = new KademliaPeer(new Destination("4hi4H1PV5KniIUTg8ZRDWi5JFC~-TdSQ3nYSIWRykBucSCbjb7XZNxt4u3H6pvSVVcCd8BdeAyfQh6isGOi9JlhP301OGMkIkLqrAYxhf293Yip4NhTkx3H7NqO0l7BNfFOSnuZ1rny8pzbgON60Y5WSvk2mn31KdgPS8Nqi50XuPpggaeiU0tuiXcCpfFgFA-r7NDVo77BWmkUsvipcpLaI7kaQ7GCMdRHgA3rQcx-p2E3QBFVxwrhNl1fZkwT12M2mfA6jVanQeu6qoOplTWA5hJ2a-Y-o7wCnwM5XzT6yNudxktPMZT3IAYm0PbJPaWU~LWY8ciaKezh~9ZgYKZYq-hStyJWadvPPuBz1nG2bh7ELQt3BCph3nk8BbkigKCtqHCbtsn-xmoj91jeF2DR1EDAp-F7a3qhponAOf~AuAxqM7nG~8JQQFQqupQpA1WAsdsZqdFicnI3wfjsnWnV8G1T813o5JYq3rqe6q35XD1NzkRDMUnXGn6wWmDMiAAAA"));
            peers3[7] = new KademliaPeer(new Destination("wU4Ohy-y3fbVIzusoZuPrY5R-UbZxdWKSiOLXuxli7WKRHaXunantDln8~CMvptQ2hGxyHXXe5PmT1zUn9ov7wF7FfGFfiPOiLJxD6F~FQsIulj3LV1TcbxxDZvG7fz-0eQV0tmEo9cQr3cvA9KcA0fTNxJp2-5PxviGBzT-QZOnNJLx5LkNjuPMQaRuFfchKHMRwWPvluQCdPStVoYjHLp3wSrkOaPt0HqGoXeAx~48iQPO9jAiEQmw8DR-tDY0ccFDqQKAdQ8BonEwjSUMGH1wecqprGZpxuVVX5kvWzV4CWJ2kWc4mDqPmjgWcwbNDQlHgk3PQ2wpNzOTduYhcA5ry6EQwXFapWP8X-NzrzyKpqjwaVp8RPtunN-TwRMi5Ykbd58RzndAaDeuTEFVQJRBdz4ffp~IdYLhCIA3ycBpAR-PEBdi~KEYkVkV3DkjvGVPLWiWogoONztz4J89V737rjCNhqBZ2Jtv9jY9oN4EnnvrtI8PsE~XhkF~M6XlAAAA"));
            peers3[8] = new KademliaPeer(new Destination("TL2VdtgRJbnJdXlSbvlfDclCaP4q8a9rTn5WlAGfJVDbI-kN4QPd1AwXpuTstaG-Ohe1jWlp2DA~~T7NPnMXI44oiJv4M8ECzSVg1vKeFYcOscNY5kC9CEusI1Y~c6b8T3TyUG5kfh3~CYPvce~TQPEQmmGjxQq8aJU~zsowJSeT1JuaZ3EpCinW0UD3~diS4pKU0hrig871CBnqj4TKcW1Jz98VbwIKvyrb5oDLxbjjUkwEspvxCoflk59hsyZHkI9QglyRb8bWa11TcxVaFLMl7s-63Z-~MMIzQj-QZ6xII0jU4Pr1oU00l7~7q7v0qwtO79ibH01t-62mFt1IzH8qR8A9YvMhaZJ-Zsqxj9UOXyXniTPe4WTe9MOoFrPlVYE9-GLnwPJLZOHdzDQRbM-31LSh-ESCQ7e8bw4SL~GQwNnptKTQSD2MwMiDNR0aOwclyE3EBQOkSrJgQDLwDguMdA1UpDHj4RH8ie6m9Vle6cytWdAti3g~LjzvYxoHAAAA"));
            peers3[9] = new KademliaPeer(new Destination("TqlmT4vLPCddmBP8zX4L6-LpcQleXiFhd8xQ3v~yQIeXtg9bWfxRkkM6T1BtHj-RZy~Qs5S53LARlxRa-J4RJEaKHv~t1SG4BaLbUNVKkLDhJXeVgvp6TNQNhV5mkjC29zUpvi2h2k8GTqqXCHmB1zZtV7bbGSLeE7tFr~iq41uoKdZf3er1osSpK2oZOn56b8B-p6sw13WxI~GzVzhjTibbfThZrrghX4S9k7Mw3l2nXHhqRRzM-Q-4nAsKhE4csNgMacPeAE~p7NrF6o7Ysuu16q8t1KAxCabS-~ORelrMFHgGAFQmYJ7pV0YbvO5PJPXGROSvSdpvmGMO86VTd0yEtC9gU~wGfKra7UkVib0gPs9ThmuLEdnPY-UvA1x9MaRrOHDuW8-LKfi2ab9coc~Vd8TghinU-0CABxDI-INewSdI600DWwPYej2pUpiLg3gFgE4P9558j6k4RPyRsWT5gNMjPDSv6F-Uhy4uHjW4SjPWHuqTszRJrY8BUX-3AAAA"));
            peers3[10] = new KademliaPeer(new Destination("FYo0U93-I2DLiWTjVuQHUumKqUVvMl5l5haYixLkA4~Y0Msv-nBUjowHMslcHky8jzNEMryamQ99vFQgWICfXH010xw9TgjGHhfbMu2ctS~g9j0bsrZZiHnhkJnYduJPoEv7I5oSmGulNUPX0QrMT43KQliLUHzHfKfuKQLJyOJz8AnOfu4Vm6G4qfvt2eSUSmGCMeHEYyIklkFQjKIw8XjhFBxmIA213bPZHmvJGRArVINb~MOh909WQtSmcBDcyDTcBHczYBHc71fj0G54TJ~B-Xznz-bIF7vnzTbvcVo1ok8uBBnqFl3tKebHUzupZoUfHXou3-8RAChWLV0mT2Q0N~gFcFYFlaqnR8JQXf-9tm1F92iVIK6yDo~DqS~J6MEeeBg~KhDMFfOAw86PBVh3n2qORUyB~cCp5VL5214PHBEbXyf8u-N6wbGt4jZboE80EGw4NnsXH4TMsE5bPk0H-IbmQtIV11jjgOSlk-XYAtfbsVlh5lPI7w4zBbdrAAAA"));
            peers3[11] = new KademliaPeer(new Destination("bPFvunZLEM3TRiHStt2tnCYDutTWc0cv3hDArd2q3bIzowc8pQgwCxkHyYgSd6i7LAVGk6c8rEoJcCmvSnVWhZShYoF-TGtFwmBW2xoqlMMnwLJ9HbEMdO-cb3i7hn~5U~SyMIaaA1-htHbaV1L6I-FP2S~-xPN8kaEv-60NjneFAIRsTEPjrUW8dWCyIW4L3It6YxjoALBElrmPKMg5F8WalfhzxTRCrHHT6UCWxhDGs3TCNNZu7qfi1JBKiS0YSEzkPN6CPeK737VR7IcjCvy9HylPB93CZ78Kj4Q~35WrDKqv3lDwFp-bb2jrvK4jruharOB06zTs9Sgdt~XOEW8RhnCM-NFlJGK1N6Y9MmOnofkMDmfvaHU0-PmNXfAq672H-3QwjfV5kRxRygipsGRd2Kr4y~2Z6oV2ju4ZgCLsiN-uMUdl5rzeIWsDBh6z9YFUxVUp71LSn3d~Y8Oamr0yYDN54g1wmqnnA4pgM7go6MZXE5UNwLd4JfbP6aTaAAAA"));
            peers3[12] = new KademliaPeer(new Destination("8KLjZzajzkFxLJDzqhnWnV3X2IaELEB~q50E2KXcmyIAVe-xfY7AmVjU~zl0T89K7v28d18leOqF7wJppJQTuJxMHEWZn1gbvEIZWTuDIhsDeFomx~tdqqpVAA~fYcuAWjvgwDuxYanPDGPBfdywA6IGVvahB-tSkDdUHIRoxpJDHSe5zRHzltvfTa2zNej--Ip1EgjClxiJA~8fDHszpEpAGkbet8Cbq0GCkI52heNFKm8L1pyNhtbjPD59-zH-5Fud40fF2BfrAWtIuxxyy7oQtqZoyb8qcdYUc9yUNJT9l-fm3XiG7N9V~c4reE5QD0HFHI0lExYzWWvGmsyoPEk49jvYD8wQeEnGRP0GpldjV05-L92UJzD1NdnbqK0wRqmTBNldOlra~gFrtWA-ZmkJLgrg5c3MEPttpHUpXprovrQLnZ61~zZw~R2VH15zi1OEFsR-HOm-VRQOqiAjsQj5daW8NgA0kavy3CsIEjHJVO7uDD7Ug~YHe6I0SeubAAAA"));
            peers3[13] = new KademliaPeer(new Destination("VjY9svhq~s912qier6zdCPUAIlsx0kP~fTs1EIMdhvvheP7a3CokyTGnTy0oSuAoKEPzyRhcUYDnzXXyfYTsyFZwjxSip2OSO2vBX8WK4Y7gdPHE1RanWCYs1zCZ45pEEJjTAU~fe~-CfW9gZX2VazshyZsCpTeh8ISudI5368bdTT66sot-OpAXuu7W4bxCm4FT6FBjleG02~C5mHSmyYgsXVQTzIpvMTnY0mIJnVFETbubGPX6VMBIUhXCbuq0PEWFePQNCltSZaf1eG55uSAR85DH5urfqp50QhAEc1TB4M8V4VzVV7GUiVZxJ5gw9ib1P1bj~GqN~qjQKQo5wg-JPNodXttAZ8Y2DVCVMyiZV4TCRazp9mrnjm6fg0L2vJm90ACibA59MK3~nMuwmWbW-EgmtaRCrVPKnPTcltkfHa-WpzYG6gx132-MFjIHQD9Srr8QKlq6DXpAKFVndnDk~zsmKq8GuExPGiYZwM9vD~C0iwXmjohGAMHw0kofAAAA"));
            peers3[14] = new KademliaPeer(new Destination("QmGC-LB83RwX8rrZT3D9yQaTh43~96fZ8LWwYanyjqNnL43uwPvhc4Y2Mog9ATY5SvdCFZC0dzBCg-AiZTcrqs6yf6Urs-Fw6ioN7pQzpUeMS~~5M7loU0Sx84os75XtkQfTnk4cLrRRYKnBoByNhBqZHt63uxlDGfyoxRh6525RSyNqfGtg7y18cl1qGj-DzIO3Auq8mZASSLNNm5lWw9L6ku7tPf4lDFuB-yWwL8~feuB3y-7WaN6ZwljVJHmms0vC8U4vpRqYoqJ-bn9oACpC5Sxq9v9vGYj38T~VuovxlnJPG8peX1ai8bqv~kmjOUR2CxhNAiWkTtiX0Es6-DGtSA~qnCdCPzfF6ELhzQ4Qnb0eHba2pbRZxuVidqFX21jL4iJYtqN7K-O2QE5XzxTYhJOnUcUWs9N2yXGFXCuPF3ujWAGrSppoNHwon9vdFLCOjTyYPkn~nu~z7PvottACja5bHdzOe1px7mkZhkAG~gLBUC6qR72pPXsOU5D5AAAA"));
            peers3[15] = new KademliaPeer(new Destination("ASmJH-imnOQURpTEe1fzXOgz-7QN7Ps~cL1HC7ZxcxrzP7UMynPI7auJG~Pv~MMRESpNkX7n0gzX0qP3avKdmBfucAN7Bc1SS7R0gKv6oqfAeLTEH1mlG-8JCHxbKXYUkr2CusmDp5gK8mPCmYAtGicWuz-nyOI7bVUJCV-kw7WcMFqHmt29iVHOVVKCs5voSTVgn34PmFaQxLnDcpdVbipJqQn8iGooj8LJoNuCg1ytJSJDUwb-T-ZuDt9aJX3e505eAYqYGrLvQFr3RkxtYTHlAOvCbeUGJ~qvt2CE7Qb7mlYsedC8maBpEYFDhXvUux6OlOaXJEJnpJKwMOBPDZpHbiyiXAlYSHmWWExCVAXICEe4NWBaSE6XBej8cvNGlK-G0xI8TWh8YlIi2hrs0TkHCd9bHWykaCeuSPzwQOuqkoRE6CTkcQNVdaJmvXJv25uMsg1VPpyfbqIasKnlaN4F3OOwZOeGGRqejjRquHz9c8c8MRJ6wbJXr4naPZdNAAAA"));
            peers3[16] = new KademliaPeer(new Destination("A4pdtGJviexN6fW8GxPhcGvUNpBcq6ta996t3HrscvhfHX11vBU0JFrn93lA7hchyNw~D4~AqQujHPA~6tbIaaF5jRubYeXSXHMxJYtW3M6U~HFdJcxuPWh9AKnQwww4mcBFgfNC2N9WyHfMBNRWoSF8r3DfzCD9ZsxFRppykFeVMySpJI3Dpp9R2jkUUWPD0uSe4zN~BEq9WgWgHVVZLT-gLOFmBsWTfQ65U0p3RMNe9NWbxjhYMHZzGoxaglhnOo3M07jZF15E~aL1b6NjMvMkY5Wbwn7~DayjEqz~ZBzeEZ4Ki4LWyMyDzyljw-jbPbbxagT~XDctpAmDvbwtezthHDZDzjhIp1r0qH~Kh5SiqNDw3E3JDW0VkRv8C~0j-SsB3keTPkpmToycmz9GRA77B~OO3BvRaJnY2WbS7wFgKv0HS8R~Wh4T9UvhkpgAaSpkTxi9TkGHtQTYJLFK8JVu0ap3yv7qvnLFKaClMNKmGZe8vl3GxprJ5mpj9h4JAAAA"));
            peers3[17] = new KademliaPeer(new Destination("69HxX803y1-hBq-bU6Y45XHZpy~BiTumtpqtcbCNsMffW494A-Re7fDkiGbYDm7vXTpEBjheHvd0pSyE4Djji8C6a-Ey0X18FdMOBWbZIRGXuPR6zM2LhNVD2yfneWJWcxAi4OCVQJTZ4mJeTw7Du688leG4eSB3vzJnO3oskV9y1UXczrxh1alDKswJ2VV2VH6VkYm2jx5M2X3ZP70eTDDt2GLEdwcZw4Dxc5f91MK~IfXykcf0fpcUWtVflvIRrzcjx8rcK1EmZtEWKOBKQPhzebZjiyyIdzuHncZsXY0ykF7o8PzVtlrNCUKIMj2wLQ3YaWmxvIf3~hj6XrH4fn9xjolaQxk1ncovdn~kGcoNxPlzS0OtwnrEPUR~WSwqNrs~5tQQF~NHg1GAOKYfaeDzpVbJntcjK0Uhr4XwGl1QW~au1x9P8INSQqtYZuhaMvF374~Q6d9CqH-4XvbpR9SNaG4vocJ1MZAQrww7Y94XT3c8yw75wcpaRnIMXtX0AAAA"));
            peers3[18] = new KademliaPeer(new Destination("4z2FbcgTnqBTotToaTdV5L37XtekdhYl9OwjLLV7NjuCwrjh8U2kbBwM4Y0FXwb4PqfkSr3UALq7Ve7RUtgDSsqM6Q07JJmND0ryDiK0YBYSMvc0nOe45nsirT70kPgTbMpAGn43hSNzMkdjk3DbTAt8cf9GLOui~ERusSYebzsPTPLWnBNJ62ojkOn01dQWS2g2sA6E4pK6ogCk7QtGbAXIT-rjC4I~RfHUcr1GgkSRh3ISc50Ne5QWaDQXnR3zOumtbl3uW31RsP4PEqSmLKec6UdUfvOUp-dU7jtHmyJjzuMv5KV8m5TMRrzWGOaLLJpSnxbE3qXG2gBEYlW4NVaaF8HQWIRq6R4TnovNN9dCkkx6Fdwhfw4fTNIrPlt1WusuZNrp8e26PzRtLSag2zTdQnDY3edxQVWNBHlOo8r9Lo15eRBT~IFvZ8Q6yMWHPxojfFvJ3HYzOOQdHg0Fgdb2hEF7A0Bt-gtHYZXRtQGTuOcBGwtnIb~cICdjuPIrAAAA"));
            peers3[19] = new KademliaPeer(new Destination("VQ2YYNsClXkMehiEuQn5xqaZO5b8iF7Vez5LwFeRW-uryUvvRM1Ktv9Hnz7SY22qmUJTwYswkUJLR0E1tov0wwrCaTMZhfJK2l6geF4wSOU4c9bFbyhfdD16rMrSJgyszakns9tDBj7dZeBwnlj-Yum9IkmWLk16qOj1nMoHxGiar~TVhZSizimkVuN3htMT7EUTmofv~1ywrgZkEFXaKv2A3w8dh3VBj4PPOE3uHVq3VQIEz4ZErk0ynnJZAxKxck3~XAboFDMwvilDcp7mZdNy6Q4h2UP6nx6FSJ5Qu~iY8-a9lSj3h4pXP00JqJHFDrHy-~B7g6-64RRMv2dDqIHgkGFqI-aAhyAhbpgX9lZs8GPOMtNXzSA3VZJBKVhEKC5885K9zfS~Kv-QvXQ8OX~0SrssjLG49foiofTZ68h9c9aXE2~efNl603HHWU8y50ZxEd3KxRcoPMwmQH87GzDPdFQl~44sBznHi91QQ5cw7bc0ZAJgqEcX2EkRfB96AAAA"));
            peers3[20] = new KademliaPeer(new Destination("TNq9lZWyU0mTvMvbtB6nvOT-QBgTYwvM0NoosHfx0sHQbCEQzId-ojhgT9RM9NpqxaKvAulHppKaDsZS5DAQs6~pJwbt12cFWc6C5pJIGMMYl0DmeTnLVmk1-CxgdGrt1TDv84qbwghcJ4FYINqw5l7efaURX1dJw5q4MCC9DE7hmI77fGXGq0WWXrgDhYThT9F061nDrGrj7YU490Q4o5q~XDLfZuMbM6bxLEECrY-yLIucq6SUrWn3GULlacGGUJ3EEHkvVedVgpUOYnPY~bEXaUCAU-R8Co5lBlrnOC7E3X4Bc~iiFKyzsCgP4WMNwuBwQXk~qZ2jgfNpSEesBIp3Wu9g1rf3s9iFIh~NqxtIXu6XhDVGylRntoAne4MIDpDqtqm38MRKsqJXEes1di53NPdiVKDJ3Y6kKMI1m8bI6ppWKa9jPrZNkjxKT0b~pUZl808aHjigdIrEdfzvLVp625LDNGbAOJ-CnJzP2rzTD8hFA97CY-lEPi-N9QlZAAAA"));
            startId = new BigInteger("93473727204000033254175814578319686612288952448482536549858183451728938339338");
            endId = new BigInteger("102977979631408937131925934912429413039310769350307229631811585815019336318945");
            bucket = new KBucket(startId, endId, depth);
            buckets.add(bucket);
            destinationMap.put(bucket, peers3);
        }
    }

    @Test
    public void testAddOrUpdate() {
        for (KBucket bucket: buckets) {
            KademliaPeer[] peers = destinationMap.get(bucket);
            
            // add a peer
            bucket.addOrUpdate(peers[0]);
            assertEquals(1, bucket.size());
            assertEquals(peers[0], bucket.iterator().next());

            // add the same peer again, verify there is no duplicate
            bucket.addOrUpdate(peers[0]);
            assertEquals(1, bucket.size());
            assertEquals(peers[0], bucket.iterator().next());

            // add a different peer
            bucket.addOrUpdate(peers[1]);
            assertEquals(2, bucket.size());
            assertEquals(peers[0], bucket.getPeer(peers[0]));
            assertEquals(peers[1], bucket.getPeer(peers[1]));
        }
    }

    @Test
    public void testSplit() throws SecurityException, IllegalArgumentException, NoSuchFieldException, IllegalAccessException {
        assertEquals("K must be an even number for this test to work.", 0, K%2);

        for (KBucket bucket: buckets) {
            KademliaPeer[] peers = destinationMap.get(bucket);
            int originalDepth = KademliaTestUtil.getDepth(bucket);
            for (int i=0; i<K; i++) {
                assertFalse(bucket.shouldSplit(peers[i]));
                bucket.addOrUpdate(peers[i]);
            }
            assertTrue(bucket.shouldSplit(peers[K]));
            KBucket newBucket = bucket.split();
            assertNotNull(newBucket);
            assertEquals(originalDepth+1, KademliaTestUtil.getDepth(bucket));
            assertEquals(originalDepth+1, KademliaTestUtil.getDepth(bucket));
            assertTrue(maxId(bucket).compareTo(minId(newBucket)) < 0);
        }
    }
    
    private BigInteger minId(KBucket bucket) {
        BigInteger minId = null;
        for (KademliaPeer peer: bucket.getPeers()) {
            BigInteger id = new BigInteger(1, peer.calculateHash().getData());
            if (minId==null || id.compareTo(minId)<0)
                minId = id;
        }
        return minId;
    }

    private BigInteger maxId(KBucket bucket) {
        BigInteger maxId = null;
        for (KademliaPeer peer: bucket.getPeers()) {
            BigInteger id = new BigInteger(1, peer.calculateHash().getData());
            if (maxId==null || id.compareTo(maxId)>0)
                maxId = id;
        }
        return maxId;
    }

    @Test
    public void testRemove() {
        for (KBucket bucket: buckets) {
            Destination[] destinations = destinationMap.get(bucket);
            assertEquals(0, bucket.size());
            bucket.addOrUpdate(new KademliaPeer(destinations[0]));
            assertEquals(1, bucket.size());
            bucket.remove(destinations[0]);
            assertEquals(0, bucket.size());
            
            bucket.addOrUpdate(new KademliaPeer(destinations[0]));
            bucket.addOrUpdate(new KademliaPeer(destinations[1]));
            bucket.addOrUpdate(new KademliaPeer(destinations[2]));
            assertEquals(3, bucket.size());
            bucket.remove(destinations[0]);
            assertEquals(2, bucket.size());
            bucket.remove(destinations[1]);
            assertEquals(1, bucket.size());
            bucket.remove(destinations[2]);
            assertEquals(0, bucket.size());
        }
    }

    @Test
    public void testIsFull() {
        for (KBucket bucket: buckets) {
            Destination[] destinations = destinationMap.get(bucket);
            for (int i=0; i<K-1; i++) {
                bucket.addOrUpdate(new KademliaPeer(destinations[i]));
                assertFalse(bucket.isFull());
            }
            bucket.addOrUpdate(new KademliaPeer(destinations[K-1]));
            assertTrue(bucket.isFull());
        }
    }
    
//    TODO test replacement cache
}