/*
 * Decompiled with CFR 0.152.
 */
package net.i2p.crypto;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.MessageDigest;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.DSAPrivateKeySpec;
import java.security.spec.DSAPublicKeySpec;
import net.i2p.I2PAppContext;
import net.i2p.crypto.CryptoConstants;
import net.i2p.crypto.SHA1;
import net.i2p.crypto.SHA1Hash;
import net.i2p.data.Hash;
import net.i2p.data.SigningPrivateKey;
import net.i2p.data.SigningPublicKey;
import net.i2p.data.SimpleDataStructure;
import net.i2p.util.Log;
import net.i2p.util.NativeBigInteger;

public class DSAEngine {
    private final Log _log;
    private final I2PAppContext _context;
    private static final boolean _useJavaLibs = false;

    public DSAEngine(I2PAppContext context) {
        this._log = context.logManager().getLog(DSAEngine.class);
        this._context = context;
    }

    public static DSAEngine getInstance() {
        return I2PAppContext.getGlobalContext().dsa();
    }

    public boolean verifySignature(net.i2p.data.Signature signature, byte[] signedData, SigningPublicKey verifyingKey) {
        boolean rv = this.verifySignature(signature, signedData, 0, signedData.length, verifyingKey);
        if (!rv && this._log.shouldLog(30)) {
            this._log.warn("TheCrypto DSA Sig Verify Fail");
        }
        return rv;
    }

    public boolean verifySignature(net.i2p.data.Signature signature, byte[] signedData, int offset, int size, SigningPublicKey verifyingKey) {
        return this.verifySignature(signature, DSAEngine.calculateHash(signedData, offset, size), verifyingKey);
    }

    public boolean verifySignature(net.i2p.data.Signature signature, InputStream in, SigningPublicKey verifyingKey) {
        return this.verifySignature(signature, this.calculateHash(in), verifyingKey);
    }

    public boolean verifySignature(net.i2p.data.Signature signature, SHA1Hash hash, SigningPublicKey verifyingKey) {
        return this.verifySig(signature, hash, verifyingKey);
    }

    public boolean verifySignature(net.i2p.data.Signature signature, Hash hash, SigningPublicKey verifyingKey) {
        return this.verifySig(signature, hash, verifyingKey);
    }

    private boolean verifySig(net.i2p.data.Signature signature, SimpleDataStructure hash, SigningPublicKey verifyingKey) {
        long start = this._context.clock().now();
        try {
            byte[] sigbytes = signature.getData();
            byte[] rbytes = new byte[20];
            byte[] sbytes = new byte[20];
            for (int x = 0; x < 40; ++x) {
                if (x < 20) {
                    rbytes[x] = sigbytes[x];
                    continue;
                }
                sbytes[x - 20] = sigbytes[x];
            }
            NativeBigInteger s = new NativeBigInteger(1, sbytes);
            NativeBigInteger r = new NativeBigInteger(1, rbytes);
            NativeBigInteger y = new NativeBigInteger(1, verifyingKey.getData());
            BigInteger w = null;
            try {
                w = s.modInverse(CryptoConstants.dsaq);
            }
            catch (ArithmeticException ae) {
                this._log.warn("modInverse() error", ae);
                return false;
            }
            byte[] data = hash.getData();
            NativeBigInteger bi = new NativeBigInteger(1, data);
            BigInteger u1 = bi.multiply(w).mod(CryptoConstants.dsaq);
            BigInteger u2 = r.multiply(w).mod(CryptoConstants.dsaq);
            BigInteger modval = CryptoConstants.dsag.modPow(u1, CryptoConstants.dsap);
            BigInteger modmulval = modval.multiply(((BigInteger)y).modPow(u2, CryptoConstants.dsap));
            BigInteger v = modmulval.mod(CryptoConstants.dsap).mod(CryptoConstants.dsaq);
            boolean ok = v.compareTo(r) == 0;
            long diff = this._context.clock().now() - start;
            if (diff > 1000L && this._log.shouldLog(30)) {
                this._log.warn("Took too long to verify the signature (" + diff + "ms)");
            }
            return ok;
        }
        catch (Exception e) {
            this._log.log(50, "Error verifying the signature", e);
            return false;
        }
    }

    public net.i2p.data.Signature sign(byte[] data, SigningPrivateKey signingKey) {
        return this.sign(data, 0, data.length, signingKey);
    }

    public net.i2p.data.Signature sign(byte[] data, int offset, int length, SigningPrivateKey signingKey) {
        if (signingKey == null || data == null || data.length <= 0) {
            return null;
        }
        SHA1Hash h = DSAEngine.calculateHash(data, offset, length);
        return this.sign(h, signingKey);
    }

    public net.i2p.data.Signature sign(InputStream in, SigningPrivateKey signingKey) {
        if (signingKey == null || in == null) {
            return null;
        }
        SHA1Hash h = this.calculateHash(in);
        return this.sign(h, signingKey);
    }

    public net.i2p.data.Signature sign(SHA1Hash hash, SigningPrivateKey signingKey) {
        return this.signIt(hash, signingKey);
    }

    public net.i2p.data.Signature sign(Hash hash, SigningPrivateKey signingKey) {
        return this.signIt(hash, signingKey);
    }

    private net.i2p.data.Signature signIt(SimpleDataStructure hash, SigningPrivateKey signingKey) {
        int i;
        BigInteger k;
        if (signingKey == null || hash == null) {
            return null;
        }
        long start = this._context.clock().now();
        net.i2p.data.Signature sig = new net.i2p.data.Signature();
        boolean ok = false;
        do {
            ok = (k = new BigInteger(160, this._context.random())).compareTo(CryptoConstants.dsaq) != 1;
        } while (!(ok = ok && !k.equals(BigInteger.ZERO)));
        BigInteger r = CryptoConstants.dsag.modPow(k, CryptoConstants.dsap).mod(CryptoConstants.dsaq);
        BigInteger kinv = k.modInverse(CryptoConstants.dsaq);
        NativeBigInteger M = new NativeBigInteger(1, hash.getData());
        NativeBigInteger x = new NativeBigInteger(1, signingKey.getData());
        BigInteger s = kinv.multiply(M.add(x.multiply(r))).mod(CryptoConstants.dsaq);
        byte[] rbytes = r.toByteArray();
        byte[] sbytes = s.toByteArray();
        byte[] out = new byte[40];
        this._context.random().harvester().feedEntropy("DSA.sign", rbytes, 0, rbytes.length);
        if (rbytes.length == 20) {
            for (i = 0; i < 20; ++i) {
                out[i] = rbytes[i];
            }
        } else if (rbytes.length == 21) {
            for (i = 0; i < 20; ++i) {
                out[i] = rbytes[i + 1];
            }
        } else {
            if (this._log.shouldLog(10)) {
                this._log.debug("Using short rbytes.length [" + rbytes.length + "]");
            }
            for (i = 0; i < rbytes.length; ++i) {
                out[i + 20 - rbytes.length] = rbytes[i];
            }
        }
        if (sbytes.length == 20) {
            for (i = 0; i < 20; ++i) {
                out[i + 20] = sbytes[i];
            }
        } else if (sbytes.length == 21) {
            for (i = 0; i < 20; ++i) {
                out[i + 20] = sbytes[i + 1];
            }
        } else {
            if (this._log.shouldLog(10)) {
                this._log.debug("Using short sbytes.length [" + sbytes.length + "]");
            }
            for (i = 0; i < sbytes.length; ++i) {
                out[i + 20 + 20 - sbytes.length] = sbytes[i];
            }
        }
        sig.setData(out);
        long diff = this._context.clock().now() - start;
        if (diff > 1000L && this._log.shouldLog(30)) {
            this._log.warn("Took too long to sign (" + diff + "ms)");
        }
        return sig;
    }

    public SHA1Hash calculateHash(InputStream in) {
        MessageDigest digest = SHA1.getInstance();
        byte[] buf = new byte[64];
        int read = 0;
        try {
            while ((read = in.read(buf)) != -1) {
                digest.update(buf, 0, read);
            }
        }
        catch (IOException ioe) {
            if (this._log.shouldLog(30)) {
                this._log.warn("Unable to hash the stream", ioe);
            }
            return null;
        }
        return new SHA1Hash(digest.digest());
    }

    public static SHA1Hash calculateHash(byte[] source, int offset, int len) {
        MessageDigest h = SHA1.getInstance();
        h.update(source, offset, len);
        byte[] digested = h.digest();
        return new SHA1Hash(digested);
    }

    private boolean altVerifySigSHA1(net.i2p.data.Signature signature, byte[] data, SigningPublicKey verifyingKey) throws GeneralSecurityException {
        Signature jsig = Signature.getInstance("SHA1withDSA");
        KeyFactory keyFact = KeyFactory.getInstance("DSA");
        DSAPublicKeySpec spec = new DSAPublicKeySpec(new NativeBigInteger(1, verifyingKey.getData()), CryptoConstants.dsap, CryptoConstants.dsaq, CryptoConstants.dsag);
        PublicKey pubKey = keyFact.generatePublic(spec);
        jsig.initVerify(pubKey);
        jsig.update(data);
        boolean rv = jsig.verify(DSAEngine.sigBytesToASN1(signature.getData()));
        return rv;
    }

    private net.i2p.data.Signature altSignSHA1(byte[] data, SigningPrivateKey privateKey) throws GeneralSecurityException {
        Signature jsig = Signature.getInstance("SHA1withDSA");
        KeyFactory keyFact = KeyFactory.getInstance("DSA");
        DSAPrivateKeySpec spec = new DSAPrivateKeySpec(new NativeBigInteger(1, privateKey.getData()), CryptoConstants.dsap, CryptoConstants.dsaq, CryptoConstants.dsag);
        PrivateKey privKey = keyFact.generatePrivate(spec);
        jsig.initSign(privKey, this._context.random());
        jsig.update(data);
        return new net.i2p.data.Signature(DSAEngine.aSN1ToSigBytes(jsig.sign()));
    }

    private static byte[] sigBytesToASN1(byte[] sig) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream(48);
        baos.write(48);
        baos.write(0);
        byte[] tmp = new byte[20];
        baos.write(2);
        System.arraycopy(sig, 0, tmp, 0, 20);
        BigInteger r = new BigInteger(1, tmp);
        byte[] b = r.toByteArray();
        baos.write(b.length);
        baos.write(b, 0, b.length);
        baos.write(2);
        System.arraycopy(sig, 20, tmp, 0, 20);
        BigInteger s = new BigInteger(1, tmp);
        b = s.toByteArray();
        baos.write(b.length);
        baos.write(b, 0, b.length);
        byte[] rv = baos.toByteArray();
        rv[1] = (byte)(rv.length - 2);
        return rv;
    }

    private static byte[] aSN1ToSigBytes(byte[] asn) {
        byte[] rv = new byte[40];
        byte rlen = asn[3];
        if ((asn[4] & 0x80) != 0) {
            throw new IllegalArgumentException("R is negative");
        }
        if (rlen > 21) {
            throw new IllegalArgumentException("R too big " + rlen);
        }
        if (rlen == 21) {
            System.arraycopy(asn, 5, rv, 0, 20);
        } else {
            System.arraycopy(asn, 4, rv, 20 - rlen, rlen);
        }
        int slenloc = 25 + rlen - 20;
        byte slen = asn[slenloc];
        if ((asn[slenloc + 1] & 0x80) != 0) {
            throw new IllegalArgumentException("S is negative");
        }
        if (slen > 21) {
            throw new IllegalArgumentException("S too big " + slen);
        }
        if (slen == 21) {
            System.arraycopy(asn, slenloc + 2, rv, 20, 20);
        } else {
            System.arraycopy(asn, slenloc + 1, rv, 40 - slen, slen);
        }
        return rv;
    }
}

