/*
 * Decompiled with CFR 0.152.
 */
package net.i2p.router.networkdb.kademlia;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import net.i2p.I2PAppContext;
import net.i2p.crypto.Blinding;
import net.i2p.crypto.EncType;
import net.i2p.crypto.SigType;
import net.i2p.data.Base64;
import net.i2p.data.BlindData;
import net.i2p.data.DataFormatException;
import net.i2p.data.DataHelper;
import net.i2p.data.Destination;
import net.i2p.data.Hash;
import net.i2p.data.PrivateKey;
import net.i2p.data.SigningPublicKey;
import net.i2p.router.RouterContext;
import net.i2p.util.Log;
import net.i2p.util.SecureFileOutputStream;

class BlindCache {
    private final RouterContext _context;
    private final ConcurrentHashMap<SigningPublicKey, BlindData> _cache;
    private final ConcurrentHashMap<SigningPublicKey, BlindData> _reverseCache;
    private final ConcurrentHashMap<Hash, BlindData> _hashCache;
    private static final String PERSIST_FILE = "router.blindcache.dat";

    public BlindCache(RouterContext ctx) {
        this._context = ctx;
        this._cache = new ConcurrentHashMap(32);
        this._reverseCache = new ConcurrentHashMap(32);
        this._hashCache = new ConcurrentHashMap(32);
    }

    public synchronized void shutdown() {
        this.store();
        this._cache.clear();
        this._reverseCache.clear();
        this._hashCache.clear();
    }

    public synchronized void startup() {
        this.load();
    }

    public Hash getHash(Destination dest) {
        Hash rv = this.getBlindedHash(dest);
        if (rv != null) {
            return rv;
        }
        return dest.getHash();
    }

    public Hash getHash(Hash h) {
        BlindData bd = this._hashCache.get(h);
        if (bd != null) {
            return bd.getBlindedHash();
        }
        return h;
    }

    public Hash getBlindedHash(Destination dest) {
        BlindData bd = this._cache.get(dest.getSigningPublicKey());
        if (bd != null) {
            return bd.getBlindedHash();
        }
        return null;
    }

    public Hash getBlindedHash(SigningPublicKey spk) {
        BlindData bd = this._cache.get(spk);
        if (bd == null) {
            bd = new BlindData((I2PAppContext)this._context, spk, Blinding.getDefaultBlindedType(spk.getType()), null);
        }
        this.addToCache(bd);
        return bd.getBlindedHash();
    }

    public void setBlinded(Destination dest, SigType blindedType, String secret) {
        SigningPublicKey spk = dest.getSigningPublicKey();
        BlindData bd = this._cache.get(spk);
        if (bd != null) {
            bd.setDestination(dest);
        } else {
            if (blindedType == null) {
                blindedType = Blinding.getDefaultBlindedType(spk.getType());
            }
            bd = new BlindData((I2PAppContext)this._context, dest, blindedType, secret);
            bd.setDestination(dest);
            this.addToCache(bd);
        }
    }

    public void setBlinded(Destination dest) {
        SigningPublicKey spk = dest.getSigningPublicKey();
        BlindData bd = this._cache.get(spk);
        if (bd != null) {
            bd.setDestination(dest);
            this._hashCache.putIfAbsent(dest.getHash(), bd);
        }
    }

    public void addToCache(BlindData bd) {
        this.storeInCache(bd);
        if (bd.getSecret() != null || bd.getAuthPrivKey() != null) {
            this.store();
        }
    }

    private void storeInCache(BlindData bd) {
        this._cache.put(bd.getUnblindedPubKey(), bd);
        this._reverseCache.put(bd.getBlindedPubKey(), bd);
        Destination dest = bd.getDestination();
        if (dest != null) {
            this._hashCache.put(dest.getHash(), bd);
        }
    }

    public BlindData getData(Destination dest) {
        BlindData rv = this.getData(dest.getSigningPublicKey());
        if (rv != null) {
            Destination d = rv.getDestination();
            if (d == null) {
                rv.setDestination(dest);
            } else if (!dest.equals(d)) {
                rv = null;
            }
        }
        return rv;
    }

    public BlindData getData(SigningPublicKey spk) {
        SigType type = spk.getType();
        if (type != SigType.EdDSA_SHA512_Ed25519 && type != SigType.RedDSA_SHA512_Ed25519) {
            return null;
        }
        return this._cache.get(spk);
    }

    public BlindData getReverseData(SigningPublicKey spk) {
        SigType type = spk.getType();
        if (type != SigType.RedDSA_SHA512_Ed25519) {
            return null;
        }
        return this._reverseCache.get(spk);
    }

    public synchronized void rollover() {
        this._reverseCache.clear();
        for (BlindData bd : this._cache.values()) {
            this._reverseCache.put(bd.getBlindedPubKey(), bd);
        }
    }

    public synchronized List<BlindData> getData() {
        ArrayList<BlindData> rv = new ArrayList<BlindData>(this._cache.size());
        rv.addAll(this._cache.values());
        return rv;
    }

    public boolean removeBlindData(SigningPublicKey spk) {
        boolean rv = false;
        BlindData bd = this._cache.remove(spk);
        if (bd != null) {
            rv = true;
            this._reverseCache.remove(bd.getBlindedPubKey());
            Hash h = bd.getDestHash();
            if (h != null) {
                this._hashCache.remove(h);
            }
            this.store();
        }
        return rv;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void load() {
        File file = new File(this._context.getConfigDir(), PERSIST_FILE);
        if (!file.exists()) {
            return;
        }
        Log log = this._context.logManager().getLog(BlindCache.class);
        int count = 0;
        BufferedReader br = null;
        try {
            br = new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(file), "ISO-8859-1"));
            String line = null;
            while ((line = br.readLine()) != null) {
                if (line.startsWith("#")) continue;
                try {
                    this.storeInCache(this.fromPersistentString(line));
                    ++count;
                }
                catch (IllegalArgumentException iae) {
                    if (!log.shouldLog(30)) continue;
                    log.warn("Error reading cache entry: " + line, iae);
                }
                catch (DataFormatException dfe) {
                    if (!log.shouldLog(30)) continue;
                    log.warn("Error reading cache entry: " + line, dfe);
                }
            }
        }
        catch (IOException ioe) {
            if (log.shouldLog(30) && file.exists()) {
                log.warn("Error reading the blinding cache file", ioe);
            }
        }
        finally {
            if (br != null) {
                try {
                    br.close();
                }
                catch (IOException iOException) {}
            }
        }
        if (log.shouldLog(20)) {
            log.info("Loaded " + count + " entries from " + file);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void store() {
        if (this._cache.isEmpty()) {
            return;
        }
        Log log = this._context.logManager().getLog(BlindCache.class);
        int count = 0;
        File file = new File(this._context.getConfigDir(), PERSIST_FILE);
        try (PrintWriter out = null;){
            out = new PrintWriter(new BufferedWriter(new OutputStreamWriter((OutputStream)new SecureFileOutputStream(file), "ISO-8859-1")));
            out.println("# Blinding cache entries. Format is: sigtype,bsigtype,authtype,time,key,[secret],[privkey],[dest]");
            for (BlindData bd : this._cache.values()) {
                out.println(BlindCache.toPersistentString(bd));
                ++count;
            }
            if (out.checkError()) {
                throw new IOException("Failed write to " + file);
            }
        }
        if (log.shouldLog(20)) {
            log.info("Stored " + count + " entries to " + file);
        }
    }

    private BlindData fromPersistentString(String line) throws DataFormatException {
        BlindData rv;
        PrivateKey privkey;
        String secret;
        long time;
        int auth;
        int ist2;
        int ist1;
        String[] ss = DataHelper.split(line, ",", 8);
        if (ss.length != 8) {
            throw new DataFormatException("bad format");
        }
        try {
            ist1 = Integer.parseInt(ss[0]);
            ist2 = Integer.parseInt(ss[1]);
            auth = Integer.parseInt(ss[2]);
            time = Long.parseLong(ss[3]);
        }
        catch (NumberFormatException nfe) {
            throw new DataFormatException("bad codes", nfe);
        }
        SigType st1 = SigType.getByCode(ist1);
        SigType st2 = SigType.getByCode(ist2);
        if (st1 == null || !st1.isAvailable() || st2 == null || !st2.isAvailable()) {
            throw new DataFormatException("bad codes");
        }
        SigningPublicKey spk = new SigningPublicKey(st1);
        spk.fromBase64(ss[4]);
        if (ss[5].length() > 0) {
            byte[] b = Base64.decode(ss[5]);
            if (b == null) {
                throw new DataFormatException("bad secret");
            }
            secret = DataHelper.getUTF8(b);
        } else {
            secret = null;
        }
        if (ss[6].length() > 0) {
            byte[] b = Base64.decode(ss[6]);
            if (b == null) {
                throw new DataFormatException("bad privkey");
            }
            privkey = new PrivateKey(EncType.ECIES_X25519, b);
        } else {
            privkey = null;
        }
        if (ss[7].length() > 0) {
            Destination dest = new Destination(ss[7]);
            if (!spk.equals(dest.getSigningPublicKey())) {
                throw new DataFormatException("spk mismatch");
            }
            rv = new BlindData((I2PAppContext)this._context, dest, st2, secret, auth, privkey);
        } else {
            rv = new BlindData((I2PAppContext)this._context, spk, st2, secret, auth, privkey);
        }
        rv.setDate(time);
        return rv;
    }

    private static String toPersistentString(BlindData bd) {
        StringBuilder buf = new StringBuilder(1024);
        SigningPublicKey spk = bd.getUnblindedPubKey();
        buf.append(spk.getType().getCode()).append(',');
        buf.append(bd.getBlindedSigType().getCode()).append(',');
        buf.append(bd.getAuthType()).append(',');
        buf.append(bd.getDate()).append(',');
        buf.append(spk.toBase64()).append(',');
        String secret = bd.getSecret();
        if (secret != null && secret.length() > 0) {
            buf.append(Base64.encode(secret));
        }
        buf.append(',');
        PrivateKey pk = bd.getAuthPrivKey();
        if (pk != null) {
            buf.append(pk.toBase64());
        }
        buf.append(',');
        Destination dest = bd.getDestination();
        if (dest != null) {
            buf.append(dest.toBase64());
        }
        return buf.toString();
    }
}

