/*
 * Decompiled with CFR 0.152.
 */
package net.i2p.crypto;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.interfaces.DSAKey;
import java.security.interfaces.DSAPrivateKey;
import java.security.interfaces.DSAPublicKey;
import java.security.interfaces.ECKey;
import java.security.interfaces.RSAKey;
import net.i2p.I2PAppContext;
import net.i2p.crypto.CryptoConstants;
import net.i2p.crypto.SHA1;
import net.i2p.crypto.SHA1Hash;
import net.i2p.crypto.SigAlgo;
import net.i2p.crypto.SigType;
import net.i2p.crypto.SigUtil;
import net.i2p.crypto.eddsa.EdDSAEngine;
import net.i2p.crypto.eddsa.EdDSAKey;
import net.i2p.data.Hash;
import net.i2p.data.SigningPrivateKey;
import net.i2p.data.SigningPublicKey;
import net.i2p.data.SimpleDataStructure;
import net.i2p.util.Log;
import net.i2p.util.NativeBigInteger;

public class DSAEngine {
    private final Log _log;
    private final I2PAppContext _context;
    private static final boolean _useJavaLibs = false;

    public DSAEngine(I2PAppContext context) {
        this._log = context.logManager().getLog(DSAEngine.class);
        this._context = context;
    }

    public static DSAEngine getInstance() {
        return I2PAppContext.getGlobalContext().dsa();
    }

    public boolean verifySignature(net.i2p.data.Signature signature, byte[] signedData, SigningPublicKey verifyingKey) {
        return this.verifySignature(signature, signedData, 0, signedData.length, verifyingKey);
    }

    public boolean verifySignature(net.i2p.data.Signature signature, byte[] signedData, int offset, int size, SigningPublicKey verifyingKey) {
        SigType type = signature.getType();
        if (type != verifyingKey.getType()) {
            throw new IllegalArgumentException("type mismatch sig=" + (Object)((Object)signature.getType()) + " key=" + (Object)((Object)verifyingKey.getType()));
        }
        if (type != SigType.DSA_SHA1) {
            try {
                boolean rv = this.altVerifySig(signature, signedData, offset, size, verifyingKey);
                if (!rv && this._log.shouldLog(30)) {
                    this._log.warn((Object)((Object)type) + " Sig Verify Fail");
                }
                return rv;
            }
            catch (GeneralSecurityException gse) {
                if (this._log.shouldLog(30)) {
                    this._log.warn((Object)((Object)type) + " Sig Verify Fail", gse);
                }
                return false;
            }
        }
        boolean rv = this.verifySignature(signature, DSAEngine.calculateHash(signedData, offset, size), verifyingKey);
        if (!rv && this._log.shouldLog(30)) {
            this._log.warn("TheCrypto DSA Sig Verify Fail");
        }
        return rv;
    }

    public boolean verifySignature(net.i2p.data.Signature signature, InputStream in, SigningPublicKey verifyingKey) {
        return this.verifySignature(signature, this.calculateHash(in), verifyingKey);
    }

    public boolean verifySignature(net.i2p.data.Signature signature, SHA1Hash hash, SigningPublicKey verifyingKey) {
        return this.verifySig(signature, hash, verifyingKey);
    }

    public boolean verifySignature(net.i2p.data.Signature signature, Hash hash, SigningPublicKey verifyingKey) {
        return this.verifySig(signature, hash, verifyingKey);
    }

    public boolean verifySignature(net.i2p.data.Signature signature, SimpleDataStructure hash, SigningPublicKey verifyingKey) {
        SigType type = signature.getType();
        if (type != verifyingKey.getType()) {
            throw new IllegalArgumentException("type mismatch sig=" + (Object)((Object)type) + " key=" + (Object)((Object)verifyingKey.getType()));
        }
        int hashlen = type.getHashLen();
        if (hash.length() != hashlen) {
            throw new IllegalArgumentException("type mismatch hash=" + hash.getClass() + " sig=" + (Object)((Object)type));
        }
        if (type == SigType.DSA_SHA1) {
            return this.verifySig(signature, hash, verifyingKey);
        }
        try {
            return this.altVerifySigRaw(signature, hash, verifyingKey);
        }
        catch (GeneralSecurityException gse) {
            if (this._log.shouldLog(30)) {
                this._log.warn((Object)((Object)type) + " Sig Verify Fail", gse);
            }
            return false;
        }
    }

    public boolean verifySignature(net.i2p.data.Signature signature, SimpleDataStructure hash, PublicKey pubKey) {
        try {
            return this.altVerifySigRaw(signature, hash, pubKey);
        }
        catch (GeneralSecurityException gse) {
            if (this._log.shouldLog(30)) {
                this._log.warn((Object)((Object)signature.getType()) + " Sig Verify Fail", gse);
            }
            return false;
        }
    }

    private boolean verifySig(net.i2p.data.Signature signature, SimpleDataStructure hash, SigningPublicKey verifyingKey) {
        if (signature.getType() != SigType.DSA_SHA1) {
            throw new IllegalArgumentException("Bad sig type " + (Object)((Object)signature.getType()));
        }
        if (verifyingKey.getType() != SigType.DSA_SHA1) {
            throw new IllegalArgumentException("Bad key type " + (Object)((Object)verifyingKey.getType()));
        }
        long start = this._context.clock().now();
        try {
            byte[] sigbytes = signature.getData();
            byte[] rbytes = new byte[20];
            byte[] sbytes = new byte[20];
            for (int x = 0; x < 40; ++x) {
                if (x < 20) {
                    rbytes[x] = sigbytes[x];
                    continue;
                }
                sbytes[x - 20] = sigbytes[x];
            }
            NativeBigInteger s = new NativeBigInteger(1, sbytes);
            NativeBigInteger r = new NativeBigInteger(1, rbytes);
            NativeBigInteger y = new NativeBigInteger(1, verifyingKey.getData());
            BigInteger w = null;
            try {
                w = s.modInverse(CryptoConstants.dsaq);
            }
            catch (ArithmeticException ae) {
                this._log.warn("modInverse() error", ae);
                return false;
            }
            byte[] data = hash.getData();
            NativeBigInteger bi = new NativeBigInteger(1, data);
            BigInteger u1 = bi.multiply(w).mod(CryptoConstants.dsaq);
            BigInteger u2 = r.multiply(w).mod(CryptoConstants.dsaq);
            BigInteger modval = CryptoConstants.dsag.modPow(u1, CryptoConstants.dsap);
            BigInteger modmulval = modval.multiply(((BigInteger)y).modPow(u2, CryptoConstants.dsap));
            BigInteger v = modmulval.mod(CryptoConstants.dsap).mod(CryptoConstants.dsaq);
            boolean ok = v.compareTo(r) == 0;
            long diff = this._context.clock().now() - start;
            if (diff > 1000L && this._log.shouldLog(30)) {
                this._log.warn("Took too long to verify the signature (" + diff + "ms)");
            }
            return ok;
        }
        catch (RuntimeException e) {
            this._log.log(50, "Error verifying the signature", e);
            return false;
        }
    }

    public net.i2p.data.Signature sign(byte[] data, SigningPrivateKey signingKey) {
        return this.sign(data, 0, data.length, signingKey);
    }

    public net.i2p.data.Signature sign(byte[] data, int offset, int length, SigningPrivateKey signingKey) {
        if (signingKey == null || data == null || data.length <= 0) {
            return null;
        }
        SigType type = signingKey.getType();
        if (type != SigType.DSA_SHA1) {
            try {
                return this.altSign(data, offset, length, signingKey);
            }
            catch (GeneralSecurityException gse) {
                if (this._log.shouldLog(40)) {
                    this._log.error((Object)((Object)type) + " Sign Fail", gse);
                }
                return null;
            }
        }
        SHA1Hash h = DSAEngine.calculateHash(data, offset, length);
        return this.sign(h, signingKey);
    }

    public net.i2p.data.Signature sign(InputStream in, SigningPrivateKey signingKey) {
        if (signingKey == null || in == null) {
            return null;
        }
        SHA1Hash h = this.calculateHash(in);
        return this.sign(h, signingKey);
    }

    public net.i2p.data.Signature sign(SHA1Hash hash, SigningPrivateKey signingKey) {
        return this.signIt(hash, signingKey);
    }

    public net.i2p.data.Signature sign(Hash hash, SigningPrivateKey signingKey) {
        return this.signIt(hash, signingKey);
    }

    public net.i2p.data.Signature sign(SimpleDataStructure hash, SigningPrivateKey signingKey) {
        SigType type = signingKey.getType();
        int hashlen = type.getHashLen();
        if (hash.length() != hashlen) {
            throw new IllegalArgumentException("type mismatch hash=" + hash.getClass() + " key=" + (Object)((Object)type));
        }
        if (type == SigType.DSA_SHA1) {
            return this.signIt(hash, signingKey);
        }
        try {
            return this.altSignRaw(hash, signingKey);
        }
        catch (GeneralSecurityException gse) {
            if (this._log.shouldLog(30)) {
                this._log.warn((Object)((Object)type) + " Sign Fail", gse);
            }
            return null;
        }
    }

    public net.i2p.data.Signature sign(SimpleDataStructure hash, PrivateKey privKey, SigType type) {
        String talgo;
        String algo = DSAEngine.getRawAlgo(privKey);
        if (!algo.equals(talgo = DSAEngine.getRawAlgo(type))) {
            throw new IllegalArgumentException("type mismatch type=" + (Object)((Object)type) + " key=" + privKey.getClass().getSimpleName());
        }
        try {
            return this.altSignRaw(algo, hash, privKey, type);
        }
        catch (GeneralSecurityException gse) {
            if (this._log.shouldLog(30)) {
                this._log.warn((Object)((Object)type) + " Sign Fail", gse);
            }
            return null;
        }
    }

    private net.i2p.data.Signature signIt(SimpleDataStructure hash, SigningPrivateKey signingKey) {
        long diff;
        int i;
        BigInteger k;
        if (signingKey == null || hash == null) {
            return null;
        }
        if (signingKey.getType() != SigType.DSA_SHA1) {
            throw new IllegalArgumentException("Bad key type " + (Object)((Object)signingKey.getType()));
        }
        long start = this._context.clock().now();
        boolean ok = false;
        do {
            ok = (k = new BigInteger(160, this._context.random())).compareTo(CryptoConstants.dsaq) != 1;
        } while (!(ok = ok && !k.equals(BigInteger.ZERO)));
        BigInteger r = CryptoConstants.dsag.modPow(k, CryptoConstants.dsap).mod(CryptoConstants.dsaq);
        BigInteger kinv = k.modInverse(CryptoConstants.dsaq);
        NativeBigInteger M = new NativeBigInteger(1, hash.getData());
        NativeBigInteger x = new NativeBigInteger(1, signingKey.getData());
        BigInteger s = kinv.multiply(M.add(x.multiply(r))).mod(CryptoConstants.dsaq);
        byte[] rbytes = r.toByteArray();
        byte[] sbytes = s.toByteArray();
        byte[] out = new byte[40];
        this._context.random().harvester().feedEntropy("DSA.sign", rbytes, 0, rbytes.length);
        if (rbytes.length == 20) {
            for (i = 0; i < 20; ++i) {
                out[i] = rbytes[i];
            }
        } else if (rbytes.length == 21) {
            for (i = 0; i < 20; ++i) {
                out[i] = rbytes[i + 1];
            }
        } else {
            if (rbytes.length > 21) {
                this._log.error("Bad R length " + rbytes.length);
                return null;
            }
            for (i = 0; i < rbytes.length; ++i) {
                out[i + 20 - rbytes.length] = rbytes[i];
            }
        }
        if (sbytes.length == 20) {
            for (i = 0; i < 20; ++i) {
                out[i + 20] = sbytes[i];
            }
        } else if (sbytes.length == 21) {
            for (i = 0; i < 20; ++i) {
                out[i + 20] = sbytes[i + 1];
            }
        } else {
            if (sbytes.length > 21) {
                this._log.error("Bad S length " + sbytes.length);
                return null;
            }
            for (i = 0; i < sbytes.length; ++i) {
                out[i + 20 + 20 - sbytes.length] = sbytes[i];
            }
        }
        if ((diff = this._context.clock().now() - start) > 1000L && this._log.shouldLog(30)) {
            this._log.warn("Took too long to sign (" + diff + "ms)");
        }
        return new net.i2p.data.Signature(out);
    }

    public SHA1Hash calculateHash(InputStream in) {
        MessageDigest digest = SHA1.getInstance();
        byte[] buf = new byte[64];
        int read = 0;
        try {
            while ((read = in.read(buf)) != -1) {
                digest.update(buf, 0, read);
            }
        }
        catch (IOException ioe) {
            if (this._log.shouldLog(30)) {
                this._log.warn("Unable to hash the stream", ioe);
            }
            return null;
        }
        return new SHA1Hash(digest.digest());
    }

    public static SHA1Hash calculateHash(byte[] source, int offset, int len) {
        MessageDigest h = SHA1.getInstance();
        h.update(source, offset, len);
        byte[] digested = h.digest();
        return new SHA1Hash(digested);
    }

    private boolean altVerifySig(net.i2p.data.Signature signature, byte[] data, int offset, int len, SigningPublicKey verifyingKey) throws GeneralSecurityException {
        SigType type = signature.getType();
        if (type != verifyingKey.getType()) {
            throw new IllegalArgumentException("type mismatch sig=" + (Object)((Object)type) + " key=" + (Object)((Object)verifyingKey.getType()));
        }
        if (type == SigType.DSA_SHA1) {
            return this.altVerifySigSHA1(signature, data, offset, len, verifyingKey);
        }
        Signature jsig = type.getBaseAlgorithm() == SigAlgo.EdDSA ? new EdDSAEngine(type.getDigestInstance()) : Signature.getInstance(type.getAlgorithmName());
        PublicKey pubKey = SigUtil.toJavaKey(verifyingKey);
        jsig.initVerify(pubKey);
        jsig.update(data, offset, len);
        boolean rv = jsig.verify(SigUtil.toJavaSig(signature));
        return rv;
    }

    private boolean altVerifySigRaw(net.i2p.data.Signature signature, SimpleDataStructure hash, SigningPublicKey verifyingKey) throws GeneralSecurityException {
        SigType type = signature.getType();
        if (type != verifyingKey.getType()) {
            throw new IllegalArgumentException("type mismatch sig=" + (Object)((Object)type) + " key=" + (Object)((Object)verifyingKey.getType()));
        }
        PublicKey pubKey = SigUtil.toJavaKey(verifyingKey);
        return this.verifySignature(signature, hash, pubKey);
    }

    private boolean altVerifySigRaw(net.i2p.data.Signature signature, SimpleDataStructure hash, PublicKey pubKey) throws GeneralSecurityException {
        SigType type = signature.getType();
        int hashlen = hash.length();
        if (type.getHashLen() != hashlen) {
            throw new IllegalArgumentException("type mismatch hash=" + hash.getClass() + " key=" + (Object)((Object)type));
        }
        String algo = DSAEngine.getRawAlgo(type);
        Signature jsig = type.getBaseAlgorithm() == SigAlgo.EdDSA ? new EdDSAEngine() : Signature.getInstance(algo);
        jsig.initVerify(pubKey);
        jsig.update(hash.getData());
        boolean rv = jsig.verify(SigUtil.toJavaSig(signature));
        return rv;
    }

    private boolean altVerifySigSHA1(net.i2p.data.Signature signature, byte[] data, int offset, int len, SigningPublicKey verifyingKey) throws GeneralSecurityException {
        Signature jsig = Signature.getInstance("SHA1withDSA");
        DSAPublicKey pubKey = SigUtil.toJavaDSAKey(verifyingKey);
        jsig.initVerify(pubKey);
        jsig.update(data, offset, len);
        boolean rv = jsig.verify(SigUtil.toJavaSig(signature));
        return rv;
    }

    private net.i2p.data.Signature altSign(byte[] data, int offset, int len, SigningPrivateKey privateKey) throws GeneralSecurityException {
        SigType type = privateKey.getType();
        if (type == SigType.DSA_SHA1) {
            return this.altSignSHA1(data, offset, len, privateKey);
        }
        Signature jsig = type.getBaseAlgorithm() == SigAlgo.EdDSA ? new EdDSAEngine(type.getDigestInstance()) : Signature.getInstance(type.getAlgorithmName());
        PrivateKey privKey = SigUtil.toJavaKey(privateKey);
        jsig.initSign(privKey, this._context.random());
        jsig.update(data, offset, len);
        return SigUtil.fromJavaSig(jsig.sign(), type);
    }

    private net.i2p.data.Signature altSignRaw(SimpleDataStructure hash, SigningPrivateKey privateKey) throws GeneralSecurityException {
        SigType type = privateKey.getType();
        String algo = DSAEngine.getRawAlgo(type);
        PrivateKey privKey = SigUtil.toJavaKey(privateKey);
        return this.altSignRaw(algo, hash, privKey, type);
    }

    private net.i2p.data.Signature altSignRaw(String algo, SimpleDataStructure hash, PrivateKey privKey, SigType type) throws GeneralSecurityException {
        int hashlen = hash.length();
        if (type.getHashLen() != hashlen) {
            throw new IllegalArgumentException("type mismatch hash=" + hash.getClass() + " key=" + (Object)((Object)type));
        }
        Signature jsig = type.getBaseAlgorithm() == SigAlgo.EdDSA ? new EdDSAEngine() : Signature.getInstance(algo);
        jsig.initSign(privKey, this._context.random());
        jsig.update(hash.getData());
        return SigUtil.fromJavaSig(jsig.sign(), type);
    }

    private net.i2p.data.Signature altSignSHA1(byte[] data, int offset, int len, SigningPrivateKey privateKey) throws GeneralSecurityException {
        Signature jsig = Signature.getInstance("SHA1withDSA");
        DSAPrivateKey privKey = SigUtil.toJavaDSAKey(privateKey);
        jsig.initSign(privKey, this._context.random());
        jsig.update(data, offset, len);
        return SigUtil.fromJavaSig(jsig.sign(), SigType.DSA_SHA1);
    }

    private static String getRawAlgo(SigType type) {
        switch (type.getBaseAlgorithm()) {
            case DSA: {
                return "NONEwithDSA";
            }
            case EC: {
                return "NONEwithECDSA";
            }
            case EdDSA: {
                return "NONEwithEdDSA";
            }
            case RSA: {
                return "NONEwithRSA";
            }
        }
        throw new UnsupportedOperationException("Raw signatures unsupported for " + (Object)((Object)type));
    }

    private static String getRawAlgo(Key key) {
        if (key instanceof DSAKey) {
            return "NONEwithDSA";
        }
        if (key instanceof ECKey) {
            return "NONEwithECDSA";
        }
        if (key instanceof EdDSAKey) {
            return "NONEwithEdDSA";
        }
        if (key instanceof RSAKey) {
            return "NONEwithRSA";
        }
        throw new UnsupportedOperationException("Raw signatures unsupported for " + key.getClass().getName());
    }
}

